/* X/lang - the eXtensible Language
 * Copyright (C) 2001 Patrick Deschenes
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* These files are distributed at http://xlang.sourceforge.net
 */

%{

#include <xltype.h>
#include <xlexpr.h>
#include <xllangc.h>

#ifndef YYDEBUG
#define	YYDEBUG	 0	/* Default to no yydebug support */
#endif

%}

%union {
  XLExpr*     expr;
  XLTypeType  xtype;
  xdouble     val_float;
  xint        val_int;
  xchar       val_string[100];
  xchar       identifier[100];
}

%token IF ELSE CLASS NATIVE_CLASS NATIVE_FIELD NATIVE_METHOD RETURN FOR WHILE CONST DYNAMIC STATIC V_TRUE V_FALSE V_NULL
%token IMPORT IFDEF OPT_PARAM ADDR_OF
%token VOID CHAR UCHAR SHORT USHORT INT UINT FLOAT DOUBLE POINTER STRING BOOL HANDLE NATIVE_STRING
%token<identifier> IDENTIFIER

%token<val_int>    V_INT
%token<val_float>  V_FLOAT
%token<val_string> V_STRING

%type<expr>  decl decl_ifdef
%type<expr>  decl_class decl_native_class decl_const decl_global decl_local decl_function decl_dynamic_function decl_import
%type<expr>  class_prop class_method class_field native_class_prop native_class_field native_class_method const_string
%type<expr>  code_line code_block code_block_or_line code_call code_invoke code_assign code_inc code_dec
%type<expr>  code_return code_while code_for
%type<expr>  code_if code_if_else 
%type<expr>  type expr identifier literal expr_array expr_access expr_cast expr_addr_of
%type<xtype> type_basic;

%%

/*************
 * Start
 */

input: 
  /* Nothing */
  | input decl
;

decl:
    decl_class             { xl_lang_decl_add ($1); }
  | decl_native_class      { xl_lang_decl_add ($1); }
  | decl_global            { xl_lang_decl_add ($1); }
  | decl_ifdef             { xl_lang_decl_add ($1); }
  | decl_const             { xl_lang_decl_add ($1); }
  | decl_function          { xl_lang_decl_add ($1); }
  | decl_import            { xl_lang_decl_add ($1); }
  | decl_dynamic_function  { xl_lang_decl_add ($1); }
  | error                  { xl_lang_c_error ("expecting a declaration"); } 
;

const_string:
    V_STRING   { $$ = xl_lang_data_string_create ($1);   }

decl_import:
    IMPORT const_string ';' { $$ = xl_lang_efnct_create1 ("import", $2); }

decl_ifdef:
    IFDEF const_string decl_const
    { $$ = xl_lang_efnct_create2 ("ifdef", $2, $3); }
;

/****************
 * Declarations
 */

class_field_list: 
  /* Nothing */
  | class_field_list class_field { xl_lang_class_add ($2); }

class_field:   
    class_prop     { $$ = $1; }
  | class_method   { $$ = $1; }
  | error          { xl_lang_c_error ("expecting a class field (property or method"); }
;

class_prop: 
    type identifier ';'
    { $$ = xl_lang_efnct_create2 ("prop", $1, $2); }
  | type identifier '[' ']' ';'
    {
      $1->type->array_type = XL_TYPE_ARRAY_TYPE_DYNAMIC;
      $1->type->array_size = 0;
      $$ = xl_lang_efnct_create2 ("prop", $1, $2);
    }
  | type error ';' { xl_lang_c_error ("expecting an identifier"); }
;

class_method: 
    type identifier 
    { xl_lang_method_create ($1, $2);  }
    '(' decl_param_list ')'
    code_block
    { $$ = xl_lang_method_finish ($7); }
  |
    STATIC type identifier 
    { xl_lang_method_static_create ($2, $3);  }
    '(' decl_param_list ')'
    code_block
    { $$ = xl_lang_method_finish ($8); }
;

decl_class: 
    CLASS identifier
    { xl_lang_class_create ($2); }
    '{' class_field_list '}' 
    { $$ = xl_lang_class_finish (); }
  | CLASS error
    { xl_lang_c_error ("Missing an identifier"); }
;

native_class_field_list: 
  /* Nothing */
  | native_class_field_list native_class_field { xl_lang_class_add ($2); }

native_class_field:   
    native_class_prop    { $$ = $1; }
  | native_class_method  { $$ = $1; }
  | error                { xl_lang_c_error ("expecting a class field (property or method"); }
;

native_class_prop: 
    NATIVE_FIELD type identifier ':' V_INT  ';'
    { $$ = xl_lang_native_field_create ($2, $3, $5); }
  | NATIVE_FIELD error ';' { xl_lang_c_error ("expecting an identifier"); }
;

native_class_method: 
  NATIVE_METHOD identifier identifier 
  { $$ = xl_lang_native_method_create ($2, $3);  }
;

decl_native_class: 
    NATIVE_CLASS identifier
    { xl_lang_native_class_create ($2); }
    '{' native_class_field_list '}' 
    { $$ = xl_lang_class_finish (); }
  | NATIVE_CLASS error
    { xl_lang_c_error ("Missing an identifier"); }
;

decl_const:
    CONST type identifier '=' expr ';'  
    { $$ = xl_lang_efnct_create3 ("const", $2, $3, $5); }
  | CONST type identifier '[' V_INT error ';'
    { xl_lang_c_error ("expecting a ']'"); }
;

decl_global:
    type identifier ';'  
    { $$ = xl_lang_efnct_create2 ("global", $1, $2); }
  | type identifier '[' ']' ';'
    {
      $1->type->array_type = XL_TYPE_ARRAY_TYPE_DYNAMIC;
      $1->type->array_size = 0;
      $$ = xl_lang_efnct_create2 ("global", $1, $2);
    }
  | type identifier '[' error ';'
    { xl_lang_c_error ("expecting a ']'"); }
  | type identifier '[' V_INT ']' ';'
    {
      $1->type->array_type = XL_TYPE_ARRAY_TYPE_FIXED;
      $1->type->array_size = $4;
      $$ = xl_lang_efnct_create2 ("global", $1, $2);
    }
  | type identifier '[' V_INT error ';'
    { xl_lang_c_error ("expecting a ']'"); }
;

decl_local: 
    type identifier ';'  
    { $$ = xl_lang_efnct_create2 ("local", $1, $2); }
  | type identifier '[' ']' ';'
    {
      $1->type->array_type = XL_TYPE_ARRAY_TYPE_DYNAMIC;
      $1->type->array_size = 0;
      $$ = xl_lang_efnct_create2 ("local", $1, $2);
    }
  | type identifier '[' error ';'
    { xl_lang_c_error ("expecting a ']'"); }
  | type identifier '[' V_INT ']' ';'
    {
      $1->type->array_type = XL_TYPE_ARRAY_TYPE_FIXED;
      $1->type->array_size = $4;
      $$ = xl_lang_efnct_create2 ("local", $1, $2);
    }
  | type identifier '[' V_INT error ';'
    { xl_lang_c_error ("expecting a ']'"); }
;

decl_param:
    type identifier 
    { xl_lang_fnct_add_param ($1, $2); }
  | type identifier '[' ']' 
    {
      $1->type->array_type = XL_TYPE_ARRAY_TYPE_DYNAMIC;
      $1->type->array_size = 0;
      xl_lang_fnct_add_param ($1, $2);
    }
  | type identifier '[' error ';'
    { xl_lang_c_error ("expecting a ']'"); }
  | type identifier '[' V_INT ']'
    {
      $1->type->array_type = XL_TYPE_ARRAY_TYPE_FIXED;
      $1->type->array_size = $4;
      xl_lang_fnct_add_param ($1, $2);
    }
  | type identifier '[' V_INT error ';'
    { xl_lang_c_error ("expecting a ']'"); }
;

decl_param_list: 
  /* Nothing */
  | decl_param
  | decl_param_list ',' decl_param
;

decl_function: 
  type identifier
  { xl_lang_fnct_create ($1, $2);  }
  '(' decl_param_list ')' 
  code_block
  { $$ = xl_lang_fnct_finish ($7); }
;

decl_param_list_opt:
    decl_param_list
  | decl_param_list ',' OPT_PARAM
    { xl_lang_fnct_opt_param (); }
;

decl_dynamic_function: 
  DYNAMIC expr const_string
  type identifier
  { xl_lang_dlfnct_create ($2, $3, $4, $5);  }
  '(' decl_param_list_opt ')' ';'
  { $$ = xl_lang_fnct_finish (NULL); }
;

/*****************
 * Basic elements
 */

type_basic:
    VOID      { $$ = XL_TYPE_TYPE_VOID; }
  | CHAR      { $$ = XL_TYPE_TYPE_S8; }
  | UCHAR     { $$ = XL_TYPE_TYPE_U8; }
  | SHORT     { $$ = XL_TYPE_TYPE_S32; }
  | USHORT    { $$ = XL_TYPE_TYPE_U16; }
  | INT       { $$ = XL_TYPE_TYPE_S32; }
  | UINT      { $$ = XL_TYPE_TYPE_U32; }
  | STRING    { $$ = XL_TYPE_TYPE_STRING; }
  | NATIVE_STRING { $$ = XL_TYPE_TYPE_STRING_NATIVE; } 
  | POINTER   { $$ = XL_TYPE_TYPE_POINTER; }
  | FLOAT     { $$ = XL_TYPE_TYPE_F32; }
  | DOUBLE    { $$ = XL_TYPE_TYPE_F64; }
  | BOOL      { $$ = XL_TYPE_TYPE_BOOL; }
;

type:   
    type_basic      { $$ = xl_lang_type_create ($1, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL); } 
  | identifier      { $$ = xl_lang_type_create (XL_TYPE_TYPE_STATIC, XL_TYPE_ARRAY_TYPE_NONE, 0, $1->identifier); 
                      x_unref ($1);
                    }
  | identifier '#'  { $$ = xl_lang_type_create (XL_TYPE_TYPE_DYNAMIC, XL_TYPE_ARRAY_TYPE_NONE, 0, $1->identifier);
                      x_unref ($1);
                    } 
;

identifier:
  IDENTIFIER 
  { $$ =  xl_expr_new_identifier ($1); }
;

literal:   
    V_STRING   { $$ = xl_lang_data_string_create ($1);   }
  | V_INT      { $$ = xl_lang_data_integer_create ($1);  }
  | V_FLOAT    { $$ = xl_lang_data_float_create ($1);    }
  | V_TRUE     { $$ = xl_lang_data_bool_create (TRUE);   }
  | V_FALSE    { $$ = xl_lang_data_bool_create (FALSE);  }
  | V_NULL     { $$ = xl_lang_data_pointer_create (NULL); }
;

/*******************
 * Expressions
 */

expr:   
    literal            { $$ = $1; }
  | '(' expr ')'       { $$ = $2; }
  | expr_cast          { $$ = $1; }
  | code_assign        { $$ = $1; }
  | code_inc           { $$ = $1; }
  | code_dec           { $$ = $1; }
  | code_call          { $$ = $1; }
  | code_invoke        { $$ = $1; }
  | expr_array         { $$ = $1; }
  | expr_access        { $$ = $1; }
  | expr_addr_of       { $$ = $1; }
  | identifier         { $$ = $1; } 
  | expr '|' expr      { $$ = xl_lang_efnct_create2 ("|", $1, $3); }
  | expr '&' expr      { $$ = xl_lang_efnct_create2 ("&", $1, $3); }
  | expr '+' expr      { $$ = xl_lang_efnct_create2 ("+", $1, $3); }
  | expr '*' expr      { $$ = xl_lang_efnct_create2 ("*", $1, $3); }
  | expr '-' expr      { $$ = xl_lang_efnct_create2 ("-", $1, $3); }
  | expr '/' expr      { $$ = xl_lang_efnct_create2 ("/", $1, $3); }
  | expr '<' '=' expr  { $$ = xl_lang_efnct_create2 ("<=", $1, $4); }
  | expr '>' '=' expr  { $$ = xl_lang_efnct_create2 (">=", $1, $4); }
  | expr '<' expr      { $$ = xl_lang_efnct_create2 ("<", $1, $3); }
  | expr '>' expr      { $$ = xl_lang_efnct_create2 (">", $1, $3); }
  | expr '=' '=' expr  { $$ = xl_lang_efnct_create2 ("==", $1, $4); }
  | expr '!' '=' expr  { $$ = xl_lang_efnct_create2 ("<>", $1, $4); }
  | expr '&' '&' expr  { $$ = xl_lang_efnct_create2 ("&&", $1, $4); }
  | expr '|' '|' expr  { $$ = xl_lang_efnct_create2 ("||", $1, $4); }
;

expr_cast:
  '(' type ')' expr { $$ = xl_lang_efnct_create2 ("cast", $2, $4); }
;

expr_array:
  expr '[' expr ']' { $$ = xl_lang_efnct_create2 ("[]", $1, $3); }
;

expr_access:
  expr '.' identifier { $$ = xl_lang_efnct_create2 (".", $1, $3); }
;

expr_addr_of:
  ADDR_OF '(' expr ')' 
  { $$ = xl_lang_efnct_create1 ("addr_of", $3); }

param_list: 
  /* Nothing */
  | expr                 { xl_lang_call_add ($1); }
  | param_list ',' expr  { xl_lang_call_add ($3); }
;

/*********************
 * Statements (CODE)
 */


code_lines: 
  /* Nothing */
  | code_lines code_line 
;

code_line:   
    decl_local      { xl_lang_block_add ($1); }
  | code_block      { xl_lang_block_add ($1); }
  | code_call ';'   { xl_lang_block_add ($1); }
  | code_invoke ';' { xl_lang_block_add ($1); }
  | code_assign ';' { xl_lang_block_add ($1); }
  | code_return     { xl_lang_block_add ($1); }
  | code_inc ';'    { xl_lang_block_add ($1); }
  | code_dec ';'    { xl_lang_block_add ($1); }
  | code_while      { xl_lang_block_add ($1); }
  | code_for        { xl_lang_block_add ($1); }
  | code_if         { xl_lang_block_add ($1); }
;

code_block: 
  '{' 
  { xl_lang_block_begin (); }  
  code_lines 
  '}'
  { $$ = xl_lang_block_end (); }
;

code_block_or_line:
    code_block { $$ = $1; }
    | { xl_lang_block_begin (); } code_line { $$ = xl_lang_block_end (); }
;

code_invoke: 
  expr '.' identifier
  { xl_lang_call_invoke_begin ($1, x_string_get_str ($3->identifier)); 
    x_unref ($3);
  }
  '(' param_list ')'
  { $$ = xl_lang_call_end (); }
;

code_call: 
  identifier
  { xl_lang_call_begin (x_string_get_str ($1->identifier)); 
    x_unref ($1);
  }
  '(' param_list ')'
  { $$ = xl_lang_call_end (); }
;

code_assign: 
    expr '=' expr  { $$ = xl_lang_efnct_create2 ("=", $1, $3); }
  | expr '=' error 
    { $$ = NULL;
      xl_lang_c_error ("= is expecting an <expr>");
      yyerrok; 
    }    
  | expr '=' expr error
    { $$ = NULL;
      xl_lang_c_error ("missing a semi-colon ';'");
      yyerrok; 
    }
;

code_inc: 
    expr '+' '+'   { $$ = xl_lang_efnct_create1 ("++", $1); }

code_dec: 
    expr '+' '+'   { $$ = xl_lang_efnct_create1 ("++", $1); }

code_return: 
    RETURN expr ';'   { $$ = xl_lang_efnct_create1 ("return", $2); }
  | RETURN ';'        { $$ = xl_lang_efnct_create ("return"); }
  | RETURN error ';'
    { $$ = NULL;
      xl_lang_c_error ("'return' is expecting an <expr>");
      yyerrok; 
    }
  | RETURN expr error
    { $$ = NULL;
      xl_lang_c_error ("missing a semi-colon ';'");
      yyerrok;
    }
;

code_for:
    FOR '(' expr ';' expr ';' expr ')' code_block_or_line
    { $$ = xl_lang_efnct_create4 ("for", $3, $5, $7, $9); }
  | FOR '(' error 
    { $$ = NULL;
      xl_lang_c_error ("'for' is expecting an <expr>");
      yyerrok; 
    }
  | FOR error 
    { $$ = NULL;
      xl_lang_c_error ("missing an open parenthesis '('");
      yyerrok; 
    }
;

code_while:  
    WHILE '(' expr ')' code_block_or_line
    { $$ = xl_lang_efnct_create2 ("while", $3, $5); }
  | WHILE '(' error 
    { $$ = NULL;
      xl_lang_c_error ("'while' is expecting an <expr>");
      yyerrok; 
    }
  | WHILE error 
    { $$ = NULL;
      xl_lang_c_error ("missing an open parenthesis '('");
      yyerrok; 
    }
;

code_if_else:
  /* empty */ { $$ = NULL; }
  | ELSE code_block_or_line { $$ = $2; }
;

code_if:  
    IF 
    '(' expr ')' code_block_or_line
    code_if_else
    { if ($5)
        $$ = xl_lang_efnct_create2 ("if", $3, $5); 
      else
        $$ = xl_lang_efnct_create1 ("if", $3);

      if ($6)
	{
	  xl_expr_add ($$, $6);
	  x_unref ($6);
	}
    }
  | IF '(' error 
    { $$ = NULL;
      xl_lang_c_error ("'if' is expecting an <expr>");
      yyerrok; 
    }
  | IF error 
    { $$ = NULL;
      xl_lang_c_error ("missing an open parenthesis '('");
      yyerrok; 
    }
;

%%

