/* XLanguage - the eXtensible Language
 * Copyright (C) 2001 Patrick Deschenes
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* These files are distributed at http://www.freesoftware.fsf.org/xlang/
 */

#include <config.h>

#include <xlbuiltin.h>
#include <xllang.h>
#include <xlclass.h>
#include <xlmain.h>
#include <xlmeth.h>

XLClass* g_xl_builtin_current_class = NULL;

#define XL_ERROR printf
#define XL_ERR_BAD_EXPR   "ERROR('%s'): '%s' expression of bad type\n"

xvoid
xl_builtin_init ()
{
  /* Declarators
   */

  xl_main_add_fnct_builtin (g_xl_main_global, "fnct", xl_builtin_fnct);
  xl_main_add_fnct_builtin (g_xl_main_global, "dlfnct", xl_builtin_dlfnct);
  xl_main_add_fnct_builtin (g_xl_main_global, "global", xl_builtin_global);
  xl_main_add_fnct_builtin (g_xl_main_global, "const", xl_builtin_const);
  xl_main_add_fnct_builtin (g_xl_main_global, "local", xl_builtin_local);
  xl_main_add_fnct_builtin (g_xl_main_global, "class", xl_builtin_class); 
  xl_main_add_fnct_builtin (g_xl_main_global, "native_class", xl_builtin_native_class);
  xl_main_add_fnct_builtin (g_xl_main_global, "native_field", xl_builtin_native_field);
  xl_main_add_fnct_builtin (g_xl_main_global, "native_method", xl_builtin_native_method);
  xl_main_add_fnct_builtin (g_xl_main_global, "meth", xl_builtin_meth);
  xl_main_add_fnct_builtin (g_xl_main_global, "meth_static", xl_builtin_meth_static);
  xl_main_add_fnct_builtin (g_xl_main_global, "prop", xl_builtin_prop);

  xl_main_add_fnct_builtin (g_xl_main_global, "cast", xl_builtin_cast);
  xl_main_add_fnct_builtin (g_xl_main_global, "addr_of", xl_builtin_addr_of);
  xl_main_add_fnct_builtin (g_xl_main_global, "set_raw", xl_builtin_set_raw);
  xl_main_add_fnct_builtin (g_xl_main_global, "str_setsize", xl_builtin_str_setsize);

  xl_main_add_fnct_builtin (g_xl_main_global, "|", xl_builtin_binary_or);
  xl_main_add_fnct_builtin (g_xl_main_global, "&", xl_builtin_binary_and);

  xl_main_add_fnct_builtin (g_xl_main_global, "+", xl_builtin_add);
  xl_main_add_fnct_builtin (g_xl_main_global, "-", xl_builtin_sub);
  xl_main_add_fnct_builtin (g_xl_main_global, "*", xl_builtin_mul);
  xl_main_add_fnct_builtin (g_xl_main_global, "/", xl_builtin_div);

  xl_main_add_fnct_builtin (g_xl_main_global, "import", xl_builtin_import);
  xl_main_add_fnct_builtin (g_xl_main_global, "ifdef", xl_builtin_ifdef);

  xl_main_add_fnct_builtin (g_xl_main_global, "new", xl_builtin_new);  
  xl_main_add_fnct_builtin (g_xl_main_global, ".", xl_builtin_access);  
  xl_main_add_fnct_builtin (g_xl_main_global, "::", xl_builtin_invoke);
  xl_main_add_fnct_builtin (g_xl_main_global, "[]", xl_builtin_array);
  xl_main_add_fnct_builtin (g_xl_main_global, "arr_getsize", xl_builtin_array_getsize);
  xl_main_add_fnct_builtin (g_xl_main_global, "arr_resize", xl_builtin_array_resize);

  xl_main_add_fnct_builtin (g_xl_main_global, "return", xl_builtin_return);
  xl_main_add_fnct_builtin (g_xl_main_global, "if", xl_builtin_if);
  xl_main_add_fnct_builtin (g_xl_main_global, "while", xl_builtin_while);
  xl_main_add_fnct_builtin (g_xl_main_global, "for", xl_builtin_for);

  xl_main_add_fnct_builtin (g_xl_main_global, "<", xl_builtin_cmp_l);
  xl_main_add_fnct_builtin (g_xl_main_global, ">", xl_builtin_cmp_g);
  xl_main_add_fnct_builtin (g_xl_main_global, "<=", xl_builtin_cmp_le);
  xl_main_add_fnct_builtin (g_xl_main_global, ">=", xl_builtin_cmp_ge);
  xl_main_add_fnct_builtin (g_xl_main_global, "<>", xl_builtin_cmp_ne);
  xl_main_add_fnct_builtin (g_xl_main_global, "==", xl_builtin_cmp_e);

  xl_main_add_fnct_builtin (g_xl_main_global, "||", xl_builtin_logic_or);
  xl_main_add_fnct_builtin (g_xl_main_global, "&&", xl_builtin_logic_and);

  xl_main_add_fnct_builtin (g_xl_main_global, "=", xl_builtin_assign);

  xl_main_add_fnct_builtin (g_xl_main_global, "++", xl_builtin_inc);
  xl_main_add_fnct_builtin (g_xl_main_global, "--", xl_builtin_dec);
  
  xl_main_add_fnct_builtin (g_xl_main_global, "@", xl_builtin_block);
}

xvoid
xl_builtin_exit ()
{
}

xvoid
xl_builtin_fnct (XLExpr* p_expr)
{
  XLFnct*  l_fnct;
  XLExpr*  l_expr;
  XString* l_function_name;
  XLType*  l_param_type;
  XString* l_param_name;
  XLField* l_field;
  XLExpr*  l_code;
  xuint     l_size;
  xuint     i;

  l_size = xl_expr_get_size (p_expr);

  l_expr = xl_expr_get (p_expr, 0);
  if (xl_expr_check (l_expr, XL_EXPR_TYPE_TYPE))
    {
      x_unref (l_expr);
    }
  else
    {
      XL_ERROR (XL_ERR_BAD_EXPR, "fnct", "return-type");
      x_unref (l_expr);
      return;
    }

  l_expr = xl_expr_get (p_expr, 1);
  if (xl_expr_check (l_expr, XL_EXPR_TYPE_IDENTIFIER))
    {
      l_function_name = x_addref (XString, l_expr->identifier);
      x_unref (l_expr);
    }
  else
    {
      XL_ERROR (XL_ERR_BAD_EXPR, "fnct", "function-name");
      x_unref (l_expr);
      return;
    }  

  l_code = xl_expr_get (p_expr, l_size - 1);
  l_fnct = xl_main_add_fnct_expr (g_xl_main_global, x_string_get_str (l_function_name), l_code);
  x_unref (l_function_name);
  x_unref (l_code);

  for (i = 0; i < ((l_size - 3) / 2); i++)
    {
      l_expr = xl_expr_get (p_expr, 2 + (i * 2));
      if (xl_expr_check (l_expr, XL_EXPR_TYPE_TYPE))
	{
	  l_param_type = x_addref (XLType, l_expr->type);
	  x_unref (l_expr);
	}
      else
	{
	  XL_ERROR (XL_ERR_BAD_EXPR, "fnct", "param-type");
	  x_unref (l_expr);
	  return;
	}

      l_expr = xl_expr_get (p_expr, 2 + (i * 2) + 1);
      if (xl_expr_check (l_expr, XL_EXPR_TYPE_IDENTIFIER))
	{
	  l_param_name = x_addref (XString, l_expr->identifier);
	  x_unref (l_expr);
	}
      else
	{
	  XL_ERROR (XL_ERR_BAD_EXPR, "fnct", "param-type");
	  x_unref (l_expr);
	  return;
	}

      l_field = xl_field_new (l_param_name, l_param_type);
      xl_fnct_add_param (l_fnct, l_field);

      x_unref (l_param_type);
      x_unref (l_param_name); 
      x_unref (l_field);
    }
}

xvoid
xl_builtin_dlfnct (XLExpr* p_expr)
{
  XLFnct*  l_fnct;
  XLExpr*  l_expr;
  XString* l_function_name;
  XLType*  l_param_type;
  XString* l_param_name;
  XString* l_library_name;
  XString* l_symbol_name;
  xptr     l_library;
  xptr     l_symbol;
  XLField* l_field;
  XLType*  l_type;
  xuint     l_size;
  xuint     i;

  l_size = xl_expr_get_size (p_expr);

  l_expr = xl_expr_get (p_expr, 0);
  xl_expr_evaluate (l_expr);
  l_library_name = xl_data_get_string (l_expr->data);
  x_unref (l_expr);

  l_expr = xl_expr_get (p_expr, 1);
  l_symbol_name = xl_data_get_string (l_expr->data);
  x_unref (l_expr);

  l_expr = xl_expr_get (p_expr, 2);
  if (xl_expr_check (l_expr, XL_EXPR_TYPE_TYPE))
    {
      l_type = x_addref (XLType, l_expr->type);
      x_unref (l_expr);
    }
  else
    {
      XL_ERROR (XL_ERR_BAD_EXPR, "fnct", "return-type");
      x_unref (l_expr);
      return;
    }

  l_expr = xl_expr_get (p_expr, 3);
  if (xl_expr_check (l_expr, XL_EXPR_TYPE_IDENTIFIER))
    {
      l_function_name = x_addref (XString, l_expr->identifier);
      x_unref (l_expr);
    }
  else
    {
      XL_ERROR (XL_ERR_BAD_EXPR, "fnct", "function-name");
      x_unref (l_expr);
      return;
    }  

  l_library = (xptr) x_dynamic_load (x_string_get_str (l_library_name));
  if (!l_library)
    {
      printf ("warning: unable to load library '%s'\n", x_string_get_str (l_library_name));
      return;
    }
  l_symbol = (xptr) x_dynamic_get_symbol (l_library, x_string_get_str (l_symbol_name));

  if (l_symbol)
    {
      l_fnct = xl_main_add_fnct_native (g_xl_main_global, x_string_get_str (l_function_name), l_symbol);
      l_fnct->ret_type = l_type;
    }
  else
    {
      printf ("warning: unkown symbol '%s'\n", x_string_get_str (l_symbol_name));
    }

  x_unref (l_library_name);
  x_unref (l_symbol_name);
  x_unref (l_function_name);

  /* This function has optional parameters
   */

  if ((l_size & 1) == 0)
    {
      l_fnct->opt_param = TRUE;
      l_size--;
    }

  for (i = 0; i < ((l_size - 5) / 2); i++)
    {
      l_expr = xl_expr_get (p_expr, 4 + (i * 2));
      if (xl_expr_check (l_expr, XL_EXPR_TYPE_TYPE))
	{
	  l_param_type = x_addref (XLType, l_expr->type);
	  x_unref (l_expr);
	}
      else
	{
	  XL_ERROR (XL_ERR_BAD_EXPR, "fnct", "param-type");
	  x_unref (l_expr);
	  return;
	}

      l_expr = xl_expr_get (p_expr, 4 + (i * 2) + 1);
      if (xl_expr_check (l_expr, XL_EXPR_TYPE_IDENTIFIER))
	{
	  l_param_name = x_addref (XString, l_expr->identifier);
	  x_unref (l_expr);
	}
      else
	{
	  XL_ERROR (XL_ERR_BAD_EXPR, "fnct", "param-type");
	  x_unref (l_expr);
	  return;
	}

      l_field = xl_field_new (l_param_name, l_param_type);
      xl_fnct_add_param (l_fnct, l_field);

      x_unref (l_param_type);
      x_unref (l_param_name); 
      x_unref (l_field);
    }
}

xvoid
xl_builtin_local (XLExpr* p_expr)
{
  XLExpr*  l_expr;
  XLType*  l_type;
  XString* l_name;
  XLData*  l_data;
  XLVar*   l_var;

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_type = l_expr->type;
  x_unref (l_expr);

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 1));
  l_name = l_expr->identifier;
  x_unref (l_expr);

  l_data = xl_data_new (l_type, NULL);
  l_var = xl_var_new (l_name, l_data);

  xl_main_add_var (g_xl_main_global, l_var, FALSE);

  x_unref (l_data);
  x_unref (l_var);
}

xvoid
xl_builtin_global (XLExpr* p_expr)
{
  XLExpr*  l_expr;
  XLType*  l_type;
  XString* l_name;
  XLData*  l_data;
  XLVar*   l_var;

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_type = l_expr->type;
  x_unref (l_expr);

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 1));
  l_name = l_expr->identifier;
  x_unref (l_expr);

  l_data = xl_data_new (l_type, NULL);
  l_var = xl_var_new (l_name, l_data);

  xl_main_add_var (g_xl_main_global, l_var, TRUE);

  x_unref (l_data);
  x_unref (l_var);
}

xvoid
xl_builtin_const (XLExpr* p_expr)
{
  XLExpr*  l_expr;
  XLType*  l_type;
  XString* l_name;
  XLData*  l_data;
  XLVar*   l_var;

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_type = l_expr->type;
  x_unref (l_expr);

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 1));
  l_name = l_expr->identifier;
  x_unref (l_expr);

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 2));
  xl_expr_evaluate (l_expr);

  l_data = xl_data_new (l_type, NULL);
  xl_data_assign (l_data, l_expr->data);

  x_unref (l_expr);

  l_var = xl_var_new (l_name, l_data);

  xl_main_add_var (g_xl_main_global, l_var, TRUE);

  x_unref (l_data);
  x_unref (l_var);
}

xvoid
xl_builtin_class (XLExpr* p_expr)
{
  XLExpr*  l_expr;
  XString* l_text;
  XLClass* l_class;
  xuint     l_size, i;

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_text = x_addref (XString, l_expr->identifier);
  l_class = xl_class_new (l_text, XL_CLASS_DEFAULT, 0);
  x_unref (l_text);
  x_unref (l_expr);

  g_xl_builtin_current_class = l_class;
  l_size = x_list_get_size (p_expr->list_expr);

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 1));
  if (l_expr->expr_type == XL_EXPR_TYPE_IDENTIFIER)
    i = 2;
  else
    i = 1;

  x_unref (l_expr);

  for (; i<l_size; i++)
    {
      l_expr = xl_expr_get (p_expr, i);
      xl_expr_evaluate (l_expr);
      x_unref (l_expr);
    }
  
  xl_main_add_class (g_xl_main_global, l_class);
  x_unref (l_class);

}

xvoid
xl_builtin_native_class (XLExpr* p_expr)
{
  XLExpr*  l_expr;
  XString* l_text;
  XLClass* l_class;
  xuint     l_size, i;

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_text = x_addref (XString, l_expr->identifier);
  l_class = xl_class_new (l_text, XL_CLASS_BINDING_PTR, 0);
  x_unref (l_text);
  x_unref (l_expr);

  g_xl_builtin_current_class = l_class;
  l_size = x_list_get_size (p_expr->list_expr);

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 1));
  if (l_expr->expr_type == XL_EXPR_TYPE_IDENTIFIER)
    i = 2;
  else
    i = 1;

  x_unref (l_expr);

  for (; i<l_size; i++)
    {
      l_expr = xl_expr_get (p_expr, i);
      xl_expr_evaluate (l_expr);
      x_unref (l_expr);
    }
  
  xl_main_add_class (g_xl_main_global, l_class);
  x_unref (l_class);

}

xvoid
xl_builtin_native_field (XLExpr* p_expr)
{
  XLType*  l_type;
  XString* l_name;
  XLField* l_field;
  XLExpr*  l_expr;
  xuint     l_offset;

  l_expr = xl_expr_get (p_expr, 2);
  xl_expr_evaluate (l_expr);
  l_offset = x_access (xl_data_get_raw (l_expr->data), 0, xuint);
  x_unref (l_expr);

  l_expr = xl_expr_get (p_expr, 0);
  l_type = x_addref (XLType, l_expr->type);
  x_unref (l_expr);

  l_expr = xl_expr_get (p_expr, 1);
  l_name = x_addref (XString, l_expr->identifier);
  x_unref (l_expr);

  l_field = xl_field_new (l_name, l_type);
  x_unref (l_name);
  x_unref (l_type);

  xl_class_add_field_offset (g_xl_builtin_current_class, l_field, l_offset);
  x_unref (l_field);
}

xvoid
xl_builtin_native_method (XLExpr* p_expr)
{
}

xvoid
xl_builtin_prop (XLExpr* p_expr)
{
  XLExpr*  l_expr;
  XLType*  l_prop_type;
  XString* l_prop_name;
  XLField* l_field;
  xuint     l_size;


  l_size = xl_expr_get_size (p_expr);

  l_expr = xl_expr_get (p_expr, 0);
  if (xl_expr_check (l_expr, XL_EXPR_TYPE_TYPE))
    {
      l_prop_type = x_addref (XLType, l_expr->type);
      x_unref (l_expr);
    }
  else
    {
      XL_ERROR (XL_ERR_BAD_EXPR, "prop", "prop-type");
      x_unref (l_expr);
      return;
    }

  l_expr = xl_expr_get (p_expr, 1);
  if (xl_expr_check (l_expr, XL_EXPR_TYPE_IDENTIFIER))
    {
      l_prop_name = x_addref (XString, l_expr->identifier);
      x_unref (l_expr);
    }
  else
    {
      XL_ERROR (XL_ERR_BAD_EXPR, "prop", "prop-name");
      x_unref (l_expr);
      return;
    }  

  l_field = xl_field_new (l_prop_name, l_prop_type);
  xl_class_add_field (g_xl_builtin_current_class, l_field);

  x_unref (l_field);
  x_unref (l_prop_type);
  x_unref (l_prop_name);
}

xvoid
xl_builtin_meth (XLExpr* p_expr)
{
  XLExpr*  l_expr;
  XLExpr*  l_subexpr;
  XString* l_name;
  XString* l_fnctname;
  XLFnct*  l_fnct;
  XLField* l_field;
  XLMeth*  l_meth;
  xuint	   l_size;
  xuint     i;

  l_size = x_list_get_size (p_expr->list_expr);
  
  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 1));
  l_name = x_string_new ();
  x_string_set (l_name, g_xl_builtin_current_class->name);
  x_string_add_str (l_name, "::");
  x_string_add (l_name, l_expr->identifier);
  l_fnctname = x_addref (XString, l_expr->identifier);

  x_unref (l_expr);

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, l_size - 1));

  l_fnct = xl_main_add_fnct_expr (g_xl_main_global, x_string_get_str (l_name), l_expr);

  /* Adding a XLMeth entry to the class
   */

  l_meth = xl_meth_new (l_fnctname, l_name, XL_METH_NORMAL);
  xl_class_add_method (g_xl_builtin_current_class, l_meth);

  x_unref (l_fnctname);
  x_unref (l_meth);


  x_unref (l_name);
  x_unref (l_expr);

  /* Adding 'self' parameter
   */
  l_name = x_string_new ();
  x_string_set_str (l_name, "self");
  l_field = xl_field_new (l_name, NULL);
  xl_fnct_add_param (l_fnct, l_field);
  x_unref (l_name);
  x_unref (l_field);

  for (i=0; i<(l_size / 2) - 1; i++)
    {
      l_subexpr = XL_EXPR (x_list_get (p_expr->list_expr, (i * 2) + 3));

      l_field = xl_field_new (l_subexpr->identifier, NULL);
      xl_fnct_add_param (l_fnct, l_field);

      x_unref (l_field);
      x_unref (l_subexpr);
    }
}

xvoid
xl_builtin_meth_static (XLExpr* p_expr)
{
  XLExpr*  l_expr;
  XLExpr*  l_subexpr;
  XString* l_name;
  XLFnct*  l_fnct;
  XLField* l_field;
  xuint	   l_size;
  xuint    i;

  l_size = x_list_get_size (p_expr->list_expr);
  
  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 1));
  l_name = x_string_new ();
  x_string_set (l_name, g_xl_builtin_current_class->name);
  x_string_add_str (l_name, "::");
  x_string_add (l_name, l_expr->identifier);
  x_unref (l_expr);

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, l_size - 1));
  l_fnct = xl_main_add_fnct_expr (g_xl_main_global, x_string_get_str (l_name), l_expr);
  x_unref (l_name);
  x_unref (l_expr);

  for (i=0; i<(l_size / 2) - 1; i++)
    {
      l_subexpr = XL_EXPR (x_list_get (p_expr->list_expr, (i * 2) + 3));

      l_field = xl_field_new (l_subexpr->identifier, NULL);
      xl_fnct_add_param (l_fnct, l_field);

      x_unref (l_field);
      x_unref (l_subexpr);
    }
}

xvoid
xl_builtin_access (XLExpr* p_expr)
{
  XLExpr*  l_expr;
  XLData*  l_data;
  XString* l_field;

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  xl_expr_evaluate (l_expr);
  l_data = l_expr->data;
  x_unref (l_expr);

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 1));
  l_field = l_expr->identifier;
  x_unref (l_expr);

  x_unref (p_expr->data);
  p_expr->data = xl_data_get_field (l_data, l_field, TRUE);
}

xvoid
xl_builtin_invoke (XLExpr* p_expr)
{
  XLClass* l_class;
  XLExpr*  l_expr;
  XLExpr*  l_subexpr;
  XLData*  l_data;
  XLMeth*  l_meth;
  XString* l_name;
  xuint    l_size, i;

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_expr->check = FALSE;
  xl_expr_evaluate (l_expr);
  l_expr->check = TRUE;

  l_data = x_addref (XLData, l_expr->data);
  if (l_data)
    {
      l_class = x_addref (XLClass, l_data->type->class);
      x_unref (l_data);
    }
  else
    l_class = xl_main_get_class (g_xl_main_global, l_expr->identifier);

  x_unref (l_expr);

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 1));
  l_meth = xl_class_get_method (l_class, l_expr->identifier);
  x_unref (l_class);
  
  if (!l_meth)
    {
      printf ("error: method '%s' from class '%s' doesn't exists\n",
	      x_string_get_str (l_expr->identifier),
	      x_string_get_str (l_class->name));
      
      exit (-1);
    }
  x_unref (l_expr);

  l_name = x_addref (XString, l_meth->fnctname);

  l_expr = xl_expr_new ();
  xl_expr_set_fnct (l_expr, l_name);
  x_unref (l_name);

  /* Adding 'self' parameter
   */

  if (l_meth->type == XL_METH_NORMAL)
    {
      l_subexpr = XL_EXPR (x_list_get (p_expr->list_expr, 0));
      xl_expr_add (l_expr, l_subexpr);
      x_unref (l_subexpr);
    }

  x_unref (l_meth);

  l_size = x_list_get_size (p_expr->list_expr);
  for (i=2; i<l_size; i++)
    {
      l_subexpr = XL_EXPR (x_list_get (p_expr->list_expr, i));
      xl_expr_add (l_expr, l_subexpr);
      x_unref (l_subexpr);
    }

  xl_expr_evaluate (l_expr);

  x_unref (p_expr->data);
  p_expr->data = x_addref (XLData, l_expr->data);

  x_unref (l_expr);
}

xvoid
xl_builtin_array (XLExpr* p_expr)
{
  XLExpr* l_expr;
  XLData* l_data;
  XLData* l_index;

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  xl_expr_evaluate (l_expr);
  l_data = x_addref (XLData, l_expr->data);
  x_unref (l_expr);

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 1));
  xl_expr_evaluate (l_expr);
  l_index = x_addref (XLData, l_expr->data);
  x_unref (l_expr);

  x_unref (p_expr->data);
  p_expr->data = xl_data_get_index (l_data, x_access (xl_data_get_raw (l_index), 0, xuint));

  x_unref (l_data);
  x_unref (l_index);
}

xvoid
xl_builtin_array_getsize (XLExpr* p_expr)
{
  XLExpr* l_expr;
  XLData* l_data;
  XLType* l_rettype;
  XLData* l_retdata;

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  xl_expr_evaluate (l_expr);
  l_data = x_addref (XLData, l_expr->data);
  x_unref (l_expr);

  l_rettype = xl_type_new (XL_TYPE_TYPE_U32, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
  l_retdata = xl_data_new (l_rettype, NULL);
  x_unref (l_rettype);

  x_access (xl_data_get_raw (l_retdata), 0, xuint) = xl_data_array_get_size (l_data);
  x_unref (l_data);

  x_unref (p_expr->data);
  p_expr->data = l_retdata;
}

xvoid
xl_builtin_array_resize (XLExpr* p_expr)
{
  XLExpr* l_expr;
  XLData* l_data;
  xuint   l_newsize;

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  xl_expr_evaluate (l_expr);
  l_data = x_addref (XLData, l_expr->data);
  x_unref (l_expr);

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 1));
  xl_expr_evaluate (l_expr);
  l_newsize = x_access (xl_data_get_raw (l_expr->data), 0, xuint);
  x_unref (l_expr);

  xl_data_array_set_size (l_data, l_newsize);

  x_unref (l_data);
}

xvoid
xl_builtin_new (XLExpr* p_expr)
{
  printf ("new\n");
}

xvoid
xl_builtin_assign  (XLExpr* p_expr)
{
  XLExpr* l_expr;
  XLData* l_src;
  XLData* l_dst;

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  xl_expr_evaluate (l_expr);
  l_dst = x_addref (XLData, l_expr->data);
  x_unref (l_expr);

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 1));  
  xl_expr_evaluate (l_expr);
  l_src = x_addref (XLData, l_expr->data);
  x_unref (l_expr);

  xl_data_assign (l_dst, l_src);

  x_unref (l_dst);
  x_unref (l_src);
}

xvoid
xl_builtin_cast  (XLExpr* p_expr)
{
  XLExpr* l_expr;
  XLData* l_src;
  XLData* l_dst;

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_dst = xl_data_new (l_expr->type, NULL);
  x_unref (l_expr);

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 1));  
  xl_expr_evaluate (l_expr);
  l_src = x_addref (XLData, l_expr->data);
  x_unref (l_expr);

  xl_data_assign (l_dst, l_src);
  x_unref (p_expr->data);
  p_expr->data = x_addref (XLData, l_dst);

  x_unref (l_dst);
  x_unref (l_src);
}

xvoid
xl_builtin_addr_of (XLExpr* p_expr)
{
  XLExpr* l_expr;
  XLType* l_type;
  XLData* l_data;

  l_type = xl_type_new (XL_TYPE_TYPE_POINTER, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
  l_data = xl_data_new (l_type, NULL);
  x_unref (l_type);

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  xl_expr_evaluate (l_expr);

  if (l_expr->data->type->type == XL_TYPE_TYPE_STRING)
    x_access (xl_data_get_raw (l_data), 0, xptr) = x_access (xl_data_get_raw (l_expr->data), 4, xptr);
  else
    x_access (xl_data_get_raw (l_data), 0, xptr) = (xptr) xl_data_get_raw (l_expr->data);

  x_unref (l_expr);

  x_unref (p_expr->data);
  p_expr->data = l_data;
}

xvoid
xl_builtin_set_raw (XLExpr* p_expr)
{
  XLExpr* l_expr;
  XLData* l_data;
  xptr    l_raw;

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 1));
  xl_expr_evaluate (l_expr);
  l_raw = x_access (xl_data_get_raw (l_expr->data), 0, xptr);
  x_unref (l_expr);

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  xl_expr_evaluate (l_expr);
  xl_data_set_raw (l_expr->data, l_raw);
  x_unref (l_expr);
}

xvoid
xl_builtin_str_setsize (XLExpr* p_expr)
{
  XLExpr*  l_expr;
  XLData*  l_data;
  xuint    l_size;
  xstr     l_str;

  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  xl_expr_evaluate (l_expr);
  l_data = l_expr->data;
  x_unref (l_expr);
 
  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 1));
  xl_expr_evaluate (l_expr);
  l_size = x_access (xl_data_get_raw (l_expr->data), 0, xuint);
  x_unref (l_expr);

  l_str = x_resize (x_access (l_data->raw, 4, xstr), l_size + 1);
  x_access (xl_data_get_raw (l_data), 4, xstr) = l_str;
 
  l_str[l_size] = '\0';
}

xvoid
xl_builtin_return (XLExpr* p_expr)
{
  XLExpr* l_expr;
  
  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 0));

  if (l_expr)
    {
      xl_expr_evaluate (l_expr);
      g_xl_fnct_return = x_addref (XLData, l_expr->data);
      x_unref (l_expr);
    }

  g_xl_fnct_halt = TRUE;
}

xvoid
xl_builtin_if (XLExpr* p_expr)
{
  XLExpr* l_test;
  XLExpr* l_true;
  XLExpr* l_false;
  
  l_test = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_true = XL_EXPR (x_list_get (p_expr->list_expr, 1));
  l_false = XL_EXPR (x_list_get (p_expr->list_expr, 2));

  xl_expr_evaluate (l_test);

  if (x_access (xl_data_get_raw (l_test->data), 0, xbool))
    {
      xl_expr_evaluate (l_true);
    }
  else
    {
      if (l_false)
	xl_expr_evaluate (l_false);
    }

  x_unref (l_test);
  x_unref (l_true);
  x_unref (l_false);
}

xvoid
xl_builtin_while (XLExpr* p_expr)
{
  XLExpr* l_test;
  XLExpr* l_code;
  
  l_test = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_code = XL_EXPR (x_list_get (p_expr->list_expr, 1));

  xl_expr_evaluate (l_test);
  while (x_access (xl_data_get_raw (l_test->data), 0, xbool))
    {
      xl_expr_evaluate (l_code);
      xl_expr_evaluate (l_test);
    }

  x_unref (l_test);
  x_unref (l_code);
}

xvoid
xl_builtin_for (XLExpr* p_expr)
{
  XLExpr* l_start;
  XLExpr* l_test;
  XLExpr* l_end;
  XLExpr* l_code;
  
  l_start = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_test = XL_EXPR (x_list_get (p_expr->list_expr, 1));
  l_end = XL_EXPR (x_list_get (p_expr->list_expr, 2));
  l_code = XL_EXPR (x_list_get (p_expr->list_expr, 3));

  xl_expr_evaluate (l_start);
  xl_expr_evaluate (l_test);
  while (x_access (xl_data_get_raw (l_test->data), 0, xbool))
    {
      xl_expr_evaluate (l_code);
      xl_expr_evaluate (l_end);
      xl_expr_evaluate (l_test);
    }

  x_unref (l_start);
  x_unref (l_test);
  x_unref (l_end);
  x_unref (l_code);
}

xvoid
xl_builtin_inc (XLExpr* p_expr)
{
  XLExpr* l_expr;
  
  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  xl_expr_evaluate (l_expr);
  x_unref (l_expr);

  x_access (xl_data_get_raw (l_expr->data), 0, xuint) = x_access (xl_data_get_raw (l_expr->data), 0, xuint) + 1;
}

xvoid
xl_builtin_dec (XLExpr* p_expr)
{
  XLExpr* l_expr;
  
  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  xl_expr_evaluate (l_expr);
  x_unref (l_expr);

  x_access (xl_data_get_raw (l_expr->data), 0, xuint) = x_access (xl_data_get_raw (l_expr->data), 0, xuint) - 1;
}

xvoid
xl_builtin_cmp_l (XLExpr* p_expr)
{
  XLExpr* l_left;
  XLExpr* l_right;
  XLType* l_type;
  XLData* l_data;

  l_left = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_right = XL_EXPR (x_list_get (p_expr->list_expr, 1));

  xl_expr_evaluate (l_left);
  xl_expr_evaluate (l_right);
 
  l_type = xl_type_new (XL_TYPE_TYPE_BOOL, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
  l_data = xl_data_new (l_type, NULL);
  x_unref (l_type);

  if (x_access (xl_data_get_raw (l_left->data), 0, xuint) < x_access (xl_data_get_raw (l_right->data), 0, xuint))
    x_access (xl_data_get_raw (l_data), 0, xbool) = TRUE;
  else
    x_access (xl_data_get_raw (l_data), 0, xbool) = FALSE;

  x_unref (l_left);
  x_unref (l_right);
  x_unref (p_expr->data);
  p_expr->data = l_data;
}

xvoid
xl_builtin_cmp_g (XLExpr* p_expr)
{
  XLExpr* l_left;
  XLExpr* l_right;
  XLType* l_type;
  XLData* l_data;

  l_left = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_right = XL_EXPR (x_list_get (p_expr->list_expr, 1));

  xl_expr_evaluate (l_left);
  xl_expr_evaluate (l_right);
 
  l_type = xl_type_new (XL_TYPE_TYPE_BOOL, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
  l_data = xl_data_new (l_type, NULL);
  x_unref (l_type);

  if (x_access (xl_data_get_raw (l_left->data), 0, xuint) > x_access (xl_data_get_raw (l_right->data), 0, xuint))
      x_access (xl_data_get_raw (l_data), 0, xbool) = TRUE;
  else
      x_access (xl_data_get_raw (l_data), 0, xbool) = FALSE;

  x_unref (l_left);
  x_unref (l_right);

  x_unref (p_expr->data);
  p_expr->data = l_data;
}

xvoid
xl_builtin_cmp_le (XLExpr* p_expr)
{
  XLExpr* l_left;
  XLExpr* l_right;
  XLType* l_type;
  XLData* l_data;

  l_left = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_right = XL_EXPR (x_list_get (p_expr->list_expr, 1));

  xl_expr_evaluate (l_left);
  xl_expr_evaluate (l_right);
 
  l_type = xl_type_new (XL_TYPE_TYPE_BOOL, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
  l_data = xl_data_new (l_type, NULL);
  x_unref (l_type);

  if (x_access (xl_data_get_raw (l_left->data), 0, xuint) <= x_access (xl_data_get_raw (l_right->data), 0, xuint))
    x_access (xl_data_get_raw (l_data), 0, xbool) = TRUE;
  else
    x_access (xl_data_get_raw (l_data), 0, xbool) = FALSE;

  x_unref (l_left);
  x_unref (l_right);
  x_unref (p_expr->data);
  p_expr->data = l_data;
}

xvoid
xl_builtin_cmp_ge (XLExpr* p_expr)
{
  XLExpr* l_left;
  XLExpr* l_right;
  XLType* l_type;
  XLData* l_data;

  l_left = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_right = XL_EXPR (x_list_get (p_expr->list_expr, 1));

  xl_expr_evaluate (l_left);
  xl_expr_evaluate (l_right);
 
  l_type = xl_type_new (XL_TYPE_TYPE_BOOL, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
  l_data = xl_data_new (l_type, NULL);
  x_unref (l_type);

  if (x_access (xl_data_get_raw (l_left->data), 0, xuint) >= x_access (xl_data_get_raw (l_right->data), 0, xuint))
    x_access (xl_data_get_raw (l_data), 0, xbool) = TRUE;
  else
    x_access (xl_data_get_raw (l_data), 0, xbool) = FALSE;

  x_unref (l_left);
  x_unref (l_right);
  x_unref (p_expr->data);
  p_expr->data = l_data;
}

xvoid
xl_builtin_cmp_ne (XLExpr* p_expr)
{
  XLExpr* l_left;
  XLExpr* l_right;
  XLType* l_type;
  XLData* l_data;

  l_left = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_right = XL_EXPR (x_list_get (p_expr->list_expr, 1));

  xl_expr_evaluate (l_left);
  xl_expr_evaluate (l_right);
 
  l_type = xl_type_new (XL_TYPE_TYPE_BOOL, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
  l_data = xl_data_new (l_type, NULL);
  x_unref (l_type);

  if (x_access (xl_data_get_raw (l_left->data), 0, xuint) != x_access (xl_data_get_raw (l_right->data), 0, xuint))
    x_access (xl_data_get_raw (l_data), 0, xbool) = TRUE;
  else
    x_access (xl_data_get_raw (l_data), 0, xbool) = FALSE;

  x_unref (l_left);
  x_unref (l_right);
  x_unref (p_expr->data);
  p_expr->data = l_data;
}

xvoid
xl_builtin_cmp_e (XLExpr* p_expr)
{
  XLExpr* l_left;
  XLExpr* l_right;
  XLType* l_type;
  XLData* l_data;

  l_left = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_right = XL_EXPR (x_list_get (p_expr->list_expr, 1));

  xl_expr_evaluate (l_left);
  xl_expr_evaluate (l_right);
 
  l_type = xl_type_new (XL_TYPE_TYPE_BOOL, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
  l_data = xl_data_new (l_type, NULL);
  x_unref (l_type);

  if (x_access (xl_data_get_raw (l_left->data), 0, xuint) == x_access (xl_data_get_raw (l_right->data), 0, xuint))
    x_access (xl_data_get_raw (l_data), 0, xbool) = TRUE;
  else
    x_access (xl_data_get_raw (l_data), 0, xbool) = FALSE;

  x_unref (l_left);
  x_unref (l_right);
  x_unref (p_expr->data);
  p_expr->data = l_data;
}

xvoid
xl_builtin_logic_or (XLExpr* p_expr)
{
  XLExpr* l_left;
  XLExpr* l_right;
  XLType* l_type;
  XLData* l_data;

  l_left = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_right = XL_EXPR (x_list_get (p_expr->list_expr, 1));

  xl_expr_evaluate (l_left);
  xl_expr_evaluate (l_right);
 
  l_type = xl_type_new (XL_TYPE_TYPE_BOOL, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
  l_data = xl_data_new (l_type, NULL);
  x_unref (l_type);

  if (x_access (xl_data_get_raw (l_left->data), 0, xuint) || x_access (xl_data_get_raw (l_right->data), 0, xuint))
    x_access (xl_data_get_raw (l_data), 0, xbool) = TRUE;
  else
    x_access (xl_data_get_raw (l_data), 0, xbool) = FALSE;

  x_unref (l_left);
  x_unref (l_right);
  x_unref (p_expr->data);
  p_expr->data = l_data;
}

xvoid
xl_builtin_logic_and (XLExpr* p_expr)
{
  XLExpr* l_left;
  XLExpr* l_right;
  XLType* l_type;
  XLData* l_data;

  l_left = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_right = XL_EXPR (x_list_get (p_expr->list_expr, 1));

  xl_expr_evaluate (l_left);
  xl_expr_evaluate (l_right);
 
  l_type = xl_type_new (XL_TYPE_TYPE_BOOL, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
  l_data = xl_data_new (l_type, NULL);
  x_unref (l_type);

  if (x_access (xl_data_get_raw (l_left->data), 0, xuint) && x_access (xl_data_get_raw (l_right->data), 0, xuint))
    x_access (xl_data_get_raw (l_data), 0, xbool) = TRUE;
  else
    x_access (xl_data_get_raw (l_data), 0, xbool) = FALSE;

  x_unref (l_left);
  x_unref (l_right);
  x_unref (p_expr->data);
  p_expr->data = l_data;
}

xvoid
xl_builtin_io_write (XLExpr* p_expr)
{
  XLExpr* l_expr;
  XLData* l_data;
  XString*l_text;
  
  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  xl_expr_evaluate (l_expr);

  l_data = l_expr->data;
  x_unref (l_expr);

  if (!l_data)
    {
      printf ("ERROR: data = NULL\n");
      return;
    }
  
  if (l_data->type->type == XL_TYPE_TYPE_STRING)
    {
      l_text = xl_data_get_string (l_data);
      printf ("%s", x_string_get_str (l_text));
      x_unref (l_text);
    }

  if (l_data->type->type == XL_TYPE_TYPE_U32)
    {
      printf ("%d", x_access (xl_data_get_raw (l_data), 0, xuint));
    }
}

xvoid
xl_builtin_block (XLExpr* p_expr)
{
  XLExpr*	 l_expr;
  xuint		 l_size;
  xuint		 i;
  
  l_size = x_list_get_size (p_expr->list_expr);
  
  for (i = 0; i < l_size; i++)
    {
      l_expr = XL_EXPR (x_list_get (p_expr->list_expr, i));
      xl_expr_evaluate (l_expr);
      x_unref (l_expr);
    }
}

xvoid
xl_builtin_binary_or (XLExpr* p_expr)
{
  XLExpr* l_left;
  XLExpr* l_right;
  XLType* l_type;
  XLData* l_data;

  l_left = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_right = XL_EXPR (x_list_get (p_expr->list_expr, 1));

  xl_expr_evaluate (l_left);
  xl_expr_evaluate (l_right);

  if (l_left->data->type->type == XL_TYPE_TYPE_U32)
    {
      l_type = xl_type_new (XL_TYPE_TYPE_U32, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
      l_data = xl_data_new (l_type, NULL);
      x_unref (l_type);
      
      x_access (xl_data_get_raw (l_data), 0, xuint) = 
	x_access (xl_data_get_raw (l_left->data), 0, xuint) | 
	x_access (xl_data_get_raw (l_right->data), 0, xuint);
    }

  x_unref (l_left);
  x_unref (l_right);
  x_unref (p_expr->data);
  p_expr->data = l_data;
}

xvoid
xl_builtin_binary_and (XLExpr* p_expr)
{
  XLExpr* l_left;
  XLExpr* l_right;
  XLType* l_type;
  XLData* l_data;

  l_left = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_right = XL_EXPR (x_list_get (p_expr->list_expr, 1));

  xl_expr_evaluate (l_left);
  xl_expr_evaluate (l_right);

  if (l_left->data->type->type == XL_TYPE_TYPE_U32)
    {
      l_type = xl_type_new (XL_TYPE_TYPE_U32, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
      l_data = xl_data_new (l_type, NULL);
      x_unref (l_type);
      
      x_access (xl_data_get_raw (l_data), 0, xuint) = 
	x_access (xl_data_get_raw (l_left->data), 0, xuint) &
	x_access (xl_data_get_raw (l_right->data), 0, xuint);
    }

  x_unref (l_left);
  x_unref (l_right);
  x_unref (p_expr->data);
  p_expr->data = l_data;
}

xvoid
xl_builtin_add (XLExpr* p_expr)
{
  XLExpr*  l_left;
  XLExpr*  l_right;
  XLType*  l_type;
  XLData*  l_data;
  XString* l_string;

  l_left = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_right = XL_EXPR (x_list_get (p_expr->list_expr, 1));

  xl_expr_evaluate (l_left);
  xl_expr_evaluate (l_right);

  if (l_left->data->type->type == XL_TYPE_TYPE_U32)
    {
      l_type = xl_type_new (XL_TYPE_TYPE_U32, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
      l_data = xl_data_new (l_type, NULL);
      x_unref (l_type);
      
      x_access (xl_data_get_raw (l_data), 0, xuint) = 
	x_access (xl_data_get_raw (l_left->data), 0, xuint) + 
	x_access (xl_data_get_raw (l_right->data), 0, xuint);
    }

  if (l_left->data->type->type == XL_TYPE_TYPE_F32)
    {
      l_type = xl_type_new (XL_TYPE_TYPE_F32, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
      l_data = xl_data_new (l_type, NULL);
      x_unref (l_type);

      x_access (xl_data_get_raw (l_data), 0, xfloat) = 
	x_access (xl_data_get_raw (l_left->data), 0, xfloat) + 
	x_access (xl_data_get_raw (l_right->data), 0, xfloat);
    }

  if (l_left->data->type->type == XL_TYPE_TYPE_STRING)
    {
      l_type = xl_type_new (XL_TYPE_TYPE_STRING, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
      l_data = xl_data_new (l_type, NULL);
      x_unref (l_type);

      l_string = x_string_new ();
      x_string_set (l_string, xl_data_get_string (l_left->data));
      x_string_add (l_string, xl_data_get_string (l_right->data));

      xl_data_set_string (l_data, l_string);
    }

  x_unref (l_left);
  x_unref (l_right);
  x_unref (p_expr->data);
  p_expr->data = l_data;
}

xvoid
xl_builtin_sub (XLExpr* p_expr)
{
  XLExpr* l_left;
  XLExpr* l_right;
  XLType* l_type;
  XLData* l_data;

  l_left = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_right = XL_EXPR (x_list_get (p_expr->list_expr, 1));

  xl_expr_evaluate (l_left);
  xl_expr_evaluate (l_right);

  if (l_left->data->type->type == XL_TYPE_TYPE_U32)
    {
      l_type = xl_type_new (XL_TYPE_TYPE_U32, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
      l_data = xl_data_new (l_type, NULL);
      x_unref (l_type);
      
      x_access (xl_data_get_raw (l_data), 0, xuint) = 
	x_access (xl_data_get_raw (l_left->data), 0, xuint) - 
	x_access (xl_data_get_raw (l_right->data), 0, xuint);
    }

  if (l_left->data->type->type == XL_TYPE_TYPE_F32)
    {

      l_type = xl_type_new (XL_TYPE_TYPE_F32, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
      l_data = xl_data_new (l_type, NULL);
      x_unref (l_type);

      x_access (xl_data_get_raw (l_data), 0, xfloat) = 
	x_access (xl_data_get_raw (l_left->data), 0, xfloat) - 
	x_access (xl_data_get_raw (l_right->data), 0, xfloat);
    }

  x_unref (l_left);
  x_unref (l_right);
  x_unref (p_expr->data);
  p_expr->data = l_data;
}

xvoid
xl_builtin_mul (XLExpr* p_expr)
{
  XLExpr* l_left;
  XLExpr* l_right;
  XLType* l_type;
  XLData* l_data;

  l_left = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_right = XL_EXPR (x_list_get (p_expr->list_expr, 1));

  xl_expr_evaluate (l_left);
  xl_expr_evaluate (l_right);

  if (l_left->data->type->type == XL_TYPE_TYPE_U32)
    {
      l_type = xl_type_new (XL_TYPE_TYPE_U32, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
      l_data = xl_data_new (l_type, NULL);
      x_unref (l_type);
      
      x_access (xl_data_get_raw (l_data), 0, xuint) = 
	x_access (xl_data_get_raw (l_left->data), 0, xuint) * 
	x_access (xl_data_get_raw (l_right->data), 0, xuint);
    }

  if (l_left->data->type->type == XL_TYPE_TYPE_F32)
    {

      l_type = xl_type_new (XL_TYPE_TYPE_F32, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
      l_data = xl_data_new (l_type, NULL);
      x_unref (l_type);

      x_access (xl_data_get_raw (l_data), 0, xfloat) = 
	x_access (xl_data_get_raw (l_left->data), 0, xfloat) *
	x_access (xl_data_get_raw (l_right->data), 0, xfloat);
    }

  x_unref (l_left);
  x_unref (l_right);
  x_unref (p_expr->data);
  p_expr->data = l_data;
}

xvoid
xl_builtin_div (XLExpr* p_expr)
{
  XLExpr* l_left;
  XLExpr* l_right;
  XLType* l_type;
  XLData* l_data;

  l_left = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  l_right = XL_EXPR (x_list_get (p_expr->list_expr, 1));

  xl_expr_evaluate (l_left);
  xl_expr_evaluate (l_right);

  if (l_left->data->type->type == XL_TYPE_TYPE_U32)
    {
      l_type = xl_type_new (XL_TYPE_TYPE_U32, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
      l_data = xl_data_new (l_type, NULL);
      x_unref (l_type);
      
      x_access (xl_data_get_raw (l_data), 0, xuint) = 
	x_access (xl_data_get_raw (l_left->data), 0, xuint) / 
	x_access (xl_data_get_raw (l_right->data), 0, xuint);
    }

  if (l_left->data->type->type == XL_TYPE_TYPE_F32)
    {

      l_type = xl_type_new (XL_TYPE_TYPE_F32, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
      l_data = xl_data_new (l_type, NULL);
      x_unref (l_type);

      x_access (xl_data_get_raw (l_data), 0, xfloat) = 
	x_access (xl_data_get_raw (l_left->data), 0, xfloat) /
	x_access (xl_data_get_raw (l_right->data), 0, xfloat);
    }

  x_unref (l_left);
  x_unref (l_right);
  x_unref (p_expr->data);
  p_expr->data = l_data;
}

xvoid
xl_builtin_import (XLExpr* p_expr)
{
  XLLang*      l_lang;
  XLExpr*      l_expr;
  XLData*      l_data;
  XString*     l_text;
  
  l_expr = XL_EXPR (x_list_get (p_expr->list_expr, 0));
  xl_expr_evaluate (l_expr);
  
  l_data = l_expr->data;
  x_unref (l_expr);
  
  if (l_data->type->type == XL_TYPE_TYPE_STRING)
    {
      l_text = xl_data_get_string (l_data);
      l_lang = xl_lang_new (x_string_get_str (l_text));
      l_expr = (*(l_lang->load)) (l_lang, l_text);

      if (l_expr)
	{
	  xl_expr_evaluate (l_expr);
	  x_unref (l_expr);
	}

      x_unref (l_lang);
      x_unref (l_text);
    }
}

xvoid
xl_builtin_ifdef (XLExpr* p_expr)
{
  XLExpr*  l_expr;
  XString* l_define;

  l_expr = xl_expr_get (p_expr, 0);
  xl_expr_evaluate (l_expr);
  l_define = xl_data_get_string (l_expr->data);
  x_unref (l_expr);

  l_expr = xl_expr_get (p_expr, 1);

  if (x_string_cmp_str (l_define, "OS_WIN32") == 0)
    {
#ifdef USE_WIN32
      xl_expr_evaluate (l_expr);
#endif
    }

  if (x_string_cmp_str (l_define, "OS_LINUX") == 0)
    {
#ifndef USE_WIN32
      xl_expr_evaluate (l_expr);
#endif
    }

  x_unref (l_define);
  x_unref (l_expr);
}

