/* XLanguage - the eXtensible Language
 * Copyright (C) 2001 Patrick Deschenes
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* These files are distributed at http://www.freesoftware.fsf.org/xlang/
 */

#include <xldata.h>
#include <xlclass.h>
#include <xlfield.h>

XLData*
xl_data_new (XLType* p_type, xptr p_ref)
{
  XLData*  self;
  
  self = x_new (XLData);
  x_object_init_object (X_OBJECT (self), xl_data_destroy);
  self->type = x_addref (XLType, p_type);
  
  if (p_ref)
    {
      self->ref = TRUE;
      self->raw = p_ref;
      return self;
    }

  self->ref = FALSE;
  self->raw = NULL; 
  xl_data_init_data (self);

  return self;
}

xvoid
xl_data_destroy (XObject* self)
{
  if (XL_DATA (self)->type->array_type == XL_TYPE_ARRAY_TYPE_NONE)
    {
      if (XL_DATA (self)->type->type == XL_TYPE_TYPE_STRING)
	{
	  if (!XL_DATA (self)->ref)
	    {
	      x_destroy (x_access (XL_DATA (self)->raw, 4, xptr));
	      x_destroy (XL_DATA (self)->raw);
	    }
	}
      else
	{
	  if (!XL_DATA (self)->ref)
	    {
	      if (XL_DATA (self)->raw)
		{
		  x_destroy (XL_DATA (self)->raw);
		}
	    }
	}
    }
  else
    {
      if (!XL_DATA (self)->ref)
	{
	  if (XL_DATA (self)->raw)
	    {
	      if (XL_DATA (self)->type->array_type == XL_TYPE_ARRAY_TYPE_DYNAMIC)
		x_destroy (x_access (XL_DATA (self)->raw, sizeof (xuint), xptr));
	    }
	  x_destroy (XL_DATA (self)->raw);
	}
    }
  x_unref (XL_DATA (self)->type);
  x_destroy (self);
}

xvoid
xl_data_init_data (XLData* self)
{
  XLClass* l_class;
  XLData*  l_data;
  XLField* l_field;
  xuint	   i, l_size;    
  xptr     l_ptr;
  xuint    l_array;

  if (self->type->array_type == XL_TYPE_ARRAY_TYPE_NONE)
    l_array = 1;
  else
    l_array = self->type->array_size;


  if (self->type->type == XL_TYPE_TYPE_STRING)
    {
      if (!self->ref)
	self->raw = x_alloc (sizeof (xuint) + sizeof (xptr));

      x_access (self->raw, 0, xuint) = 0;
      l_ptr = x_alloc (1);
      x_access (self->raw, 4, xptr) = l_ptr;
      x_access (l_ptr, 0, xchar) = '\0';
    }

  if (self->type->type == XL_TYPE_TYPE_U16)
    {
      if (!self->ref)
	self->raw = x_alloc (sizeof (xushort));
      x_access (self->raw, 0, xushort) = 0;
    }

  if (self->type->type == XL_TYPE_TYPE_U32)
    {
      if (self->type->array_type == XL_TYPE_ARRAY_TYPE_NONE)
	{
	  if (!self->ref)
	    self->raw = x_alloc (sizeof (xuint));

	  x_access (self->raw, 0, xuint) = 0;
	}
      if (self->type->array_type == XL_TYPE_ARRAY_TYPE_FIXED)
	{
	  self->raw = x_alloc (sizeof (xuint) * l_array);
	  for (i=0; i<l_array; i++)
	    x_access (self->raw, i*sizeof(xuint), xuint) = 0;
	}
      if (self->type->array_type == XL_TYPE_ARRAY_TYPE_DYNAMIC)
	{
	  self->raw = x_alloc (sizeof (xuint) + sizeof (xptr));
	  x_access (self->raw, 0, xuint) = 0;
	  x_access (self->raw, sizeof (xuint), xptr) = x_alloc (0);
	}
    }

  if (self->type->type == XL_TYPE_TYPE_S32)
    {
      self->raw = x_alloc (sizeof (xint));
      x_access (self->raw, 0, xint) = 0;
    }

  if (self->type->type == XL_TYPE_TYPE_F32)
    {
      self->raw = x_alloc (sizeof (xfloat));
      x_access (self->raw, 0, xfloat) = 0.0;
    }

  if (self->type->type == XL_TYPE_TYPE_POINTER)
    {
      self->raw = x_alloc (sizeof (xptr) * l_array);
      for (i=0; i<l_array; i++)
	x_access (self->raw, i*sizeof(xptr), xptr) = NULL;
    }

  if (self->type->type == XL_TYPE_TYPE_STRING_NATIVE)
    {
      self->raw = x_alloc (sizeof (xptr) * l_array);
      for (i=0; i<l_array; i++)
	x_access (self->raw, i*sizeof(xptr), xptr) = NULL;
    }

  if (self->type->type == XL_TYPE_TYPE_BOOL)
    {
      self->raw = x_alloc (sizeof (xbool));
      x_access (self->raw, 0, xbool) = FALSE;
    }

  if (self->type->type == XL_TYPE_TYPE_STATIC)
    {
      l_class = xl_type_get_class (self->type);
      switch (l_class->type)
	{
	case XL_CLASS_BINDING_PTR:
	  if (!self->ref)
	    self->raw = x_alloc (sizeof (xptr));
	  
	  x_access (self->raw, 0, xptr) = NULL;

	  break;
	case XL_CLASS_DEFAULT:
	  if (self->type->array_type == XL_TYPE_ARRAY_TYPE_NONE)
	    {
	      if (!self->ref)
		self->raw = x_alloc (xl_class_get_size (l_class));
	      
	      /* We loop in all fields
	       */
	      l_size = x_list_get_size (l_class->list_fields);      
	      for (i = 0; i < l_size; i++)
		{
		  l_field = XL_FIELD (x_list_get (l_class->list_fields, i));
		  l_data = xl_data_get_field (self, l_field->name, TRUE);
		  xl_data_init_data (l_data);
		  
		  x_unref (l_data);
		  x_unref (l_field);
		}
	    }

	  if (self->type->array_type == XL_TYPE_ARRAY_TYPE_FIXED)
	    {
	      if (!self->ref)
		{
		  self->raw = x_alloc (sizeof (xuint) + sizeof (xptr));
		  x_access (self->raw, 0, xuint) = self->type->array_size;
		  x_access (self->raw, sizeof (xuint), xptr) = x_alloc (0);
		}
	    }

	  if (self->type->array_type == XL_TYPE_ARRAY_TYPE_DYNAMIC)
	    {
	      if (!self->ref)
		{
		  self->raw = x_alloc (sizeof (xuint) + sizeof (xptr));
		  x_access (self->raw, 0, xuint) = 0;
		  x_access (self->raw, sizeof (xuint), xptr) = x_alloc (0);
		}
	    }
	  break;
	}
    }

  if (self->type->type == XL_TYPE_TYPE_DYNAMIC)
    {
      /* We loop in all fields
       */

      l_class = xl_type_get_class (self->type);
      l_size = x_list_get_size (l_class->list_fields);      
      for (i = 0; i < l_size; i++)
	{
	  l_field = XL_FIELD (x_list_get (l_class->list_fields, i));
	  l_data = xl_data_get_field (self, l_field->name, TRUE);
	  xl_data_init_data (l_data);
	  
	  x_unref (l_data);
	  x_unref (l_field);
	}      
    }
}

xvoid
xl_data_set_type (XLData* self, XLType* p_type)
{
}

XString*
xl_data_get_string (XLData* self)
{
  XString* l_text;

  l_text = x_string_new ();
  x_string_set_str (l_text, x_access (self->raw, 4, xstr));

  return l_text;
}

xvoid
xl_data_set_string (XLData* self, XString* p_value)
{
  xuint l_size;

  l_size = x_string_get_size (p_value);

  x_access (self->raw, 4, xstr) = x_resize (x_access (self->raw, 4, xstr), l_size + 1);
  sprintf (x_access (self->raw, 4, xstr), "%s", x_string_get_str (p_value));
}

xptr
xl_data_get_raw (XLData* self)
{
  return self->raw;
}

xvoid
xl_data_assign (XLData* self, XLData* p_src)
{
  XString* l_text;
  XLClass* l_class;
  xchar    l_str[1000];

  if (!p_src)
    {
      printf ("ASSIGN ERROR: Source not available\n");
      return;
    }

  if (self->type->type == XL_TYPE_TYPE_STRING)
    {
      if (p_src->type->type == XL_TYPE_TYPE_STRING)
	{
	  l_text = xl_data_get_string (p_src);
	  xl_data_set_string (self, l_text);
	  x_unref (l_text);
	}
      if (p_src->type->type == XL_TYPE_TYPE_U32)
	{
	  sprintf (l_str, "%d", x_access (p_src->raw, 0, xuint));
	  l_text = x_string_new ();
	  x_string_set_str (l_text, l_str);
	  xl_data_set_string (self, l_text);
	  x_unref (l_text);
	}
      if (p_src->type->type == XL_TYPE_TYPE_STRING_NATIVE)
	{
	  sprintf (l_str, "%s", x_access (p_src->raw, 0, xstr));
	  l_text = x_string_new ();
	  x_string_set_str (l_text, l_str);
	  xl_data_set_string (self, l_text);
	  x_unref (l_text);
	}
      if (p_src->type->type == XL_TYPE_TYPE_F32)
	{
	  sprintf (l_str, "%f", x_access (p_src->raw, 0, xfloat));
	  l_text = x_string_new ();
	  x_string_set_str (l_text, l_str);
	  xl_data_set_string (self, l_text);
	  x_unref (l_text);
	}
      if (p_src->type->type == XL_TYPE_TYPE_F64)
	{
	  sprintf (l_str, "%f", x_access (p_src->raw, 0, xdouble));
	  l_text = x_string_new ();
	  x_string_set_str (l_text, l_str);
	  xl_data_set_string (self, l_text);
	  x_unref (l_text);
	}
    }

  if (self->type->type == XL_TYPE_TYPE_STRING_NATIVE)
    {
      switch (p_src->type->type)
	{
	case XL_TYPE_TYPE_STRING:
	  x_access (self->raw, 0, xptr) = x_access (p_src->raw, 4, xptr);
	  break;
	default:
	  x_access (self->raw, 0, xptr) = x_access (p_src->raw, 0, xptr);
	  break;
	}
    }

  if (self->type->type == XL_TYPE_TYPE_U32)
    {
      x_access (self->raw, 0, xuint) = x_access (p_src->raw, 0, xuint);
    }

  if (self->type->type == XL_TYPE_TYPE_U16)
    {
      x_access (self->raw, 0, xushort) = x_access (p_src->raw, 0, xushort);
    }

  if (self->type->type == XL_TYPE_TYPE_F32)
    {
      x_access (self->raw, 0, xfloat) = x_access (p_src->raw, 0, xfloat);
    }

  if (self->type->type == XL_TYPE_TYPE_POINTER)
    {
      x_access (self->raw, 0, xptr) = x_access (p_src->raw, 0, xptr);
    }

  l_class = xl_type_get_class (self->type);
  if (l_class)
    {
      switch (l_class->type)
	{
	case XL_CLASS_BINDING_PTR:
	  x_access (self->raw, 0, xptr) = x_access (p_src->raw, 0, xptr);
	  break;
	}
    }
}

XLData*
xl_data_copy (XLData* self)
{
  return self;
}

XLData*
xl_data_addref_field (XLData* self, XLField* p_field)
{
  return NULL;
}

XLData*
xl_data_convert (XLData* self, XLType* p_type)
{
  return NULL;
}

XLData*
xl_data_get_index (XLData* self, xuint p_index)
{
  XLData* l_data;
  XLType* l_type;
  xptr    l_raw;
  xuint   l_size;

  if (self->type->array_type == XL_TYPE_ARRAY_TYPE_FIXED)
    {
      l_type = xl_type_new (self->type->type, XL_TYPE_ARRAY_TYPE_NONE, 0, self->type->classname);

      l_size = xl_type_get_size (l_type);
      l_data = xl_data_new (l_type, &x_access (self->raw, (p_index*l_size), xptr));
      x_unref (l_type);
    }
  if (self->type->array_type == XL_TYPE_ARRAY_TYPE_DYNAMIC)
    {
      l_type = xl_type_new (self->type->type, XL_TYPE_ARRAY_TYPE_NONE, 0, self->type->classname);
      l_raw = x_access (self->raw, sizeof (xuint), xptr);

      l_size = xl_type_get_size (l_type);
      l_data = xl_data_new (l_type, &x_access (l_raw, (p_index*l_size), xptr));
      x_unref (l_type);
    }

  return l_data;
}

xuint
xl_data_array_get_size (XLData* self)
{
  x_assert (self != NULL);

  if (self->type->array_type == XL_TYPE_ARRAY_TYPE_FIXED)
    {
      return self->type->array_size;
    }
  if (self->type->array_type == XL_TYPE_ARRAY_TYPE_DYNAMIC)
    {
      return x_access (self->raw, 0, xuint);
    }
  return 0;
}

xvoid
xl_data_array_set_size (XLData* self, xuint p_newsize)
{
  XLData* l_data;
  xuint   l_oldsize;
  xuint   i;

  x_assert (self != NULL);

  if (self->type->array_type == XL_TYPE_ARRAY_TYPE_FIXED)
    {
      printf ("error: unable to resize a fixed array\n");
    }
  if (self->type->array_type == XL_TYPE_ARRAY_TYPE_DYNAMIC)
    {
      l_oldsize = x_access (self->raw, 0, xuint);

      x_access (self->raw, 0, xuint) = p_newsize;
      x_access (self->raw, sizeof (xuint), xptr) = 
	x_resize (x_access (self->raw, sizeof (xuint), xptr), 
		  p_newsize * (xl_type_get_size (self->type)));

      
      if (p_newsize > l_oldsize)
	{
	  for (i=l_oldsize; i<p_newsize; i++)
	    {
	      l_data = xl_data_get_index (self, i);
	      xl_data_init_data (l_data);
	      x_unref (l_data);
	    }
	}
    }
}

XLData*
xl_data_get_field (XLData* self, XString* p_fieldname, xbool p_force)
{
  XLType*  l_type;
  XLData*  l_data;
  XLField* l_field;
  XLClass* l_class;
  xptr     l_raw;

  l_field = xl_class_get_field (xl_type_get_class (self->type), p_fieldname);

  if (!l_field)
    {
      if (p_force)
	{
	  printf ("error: field '%s' not available\n", x_string_get_str (p_fieldname));
	  exit (-1);
	}
      else
	return NULL;
    }

  l_type = l_field->type;
  l_class = xl_type_get_class (self->type);

  if (l_class)
    {
      switch (self->type->array_type)
	{
	case XL_TYPE_ARRAY_TYPE_NONE:
	  l_raw = self->raw;
	  break;
	case XL_TYPE_ARRAY_TYPE_FIXED:
	  break;
	case XL_TYPE_ARRAY_TYPE_DYNAMIC:
	  l_raw = x_access (self->raw, sizeof (xuint), xptr);
	  break;
	}

      switch (self->type->class->type)
	{
	case XL_CLASS_BINDING_PTR:
	  l_data = xl_data_new (l_type, ((xuchar*) x_access (l_raw, 0, xptr)) + l_field->index);
	  break;
	default:
	  l_data = xl_data_new (l_type, ((xuchar*) l_raw) + l_field->index);
	  break;
	}
    }
  else
    l_data = xl_data_new (l_type, ((xuchar*) self->raw) + l_field->index);

  x_unref (l_field);

  return l_data;
}

xptr
xl_data_get_ptr (XLData* self)
{
  return NULL;
}

xvoid
xl_data_set_raw (XLData* self, xptr p_raw)
{
  if (self->type->array_type != XL_TYPE_ARRAY_TYPE_NONE)
    {
      self->raw = p_raw;
    }
}
