/* X Language - the eXtensible Language
 * Copyright (C) 2001 Xiong PuXiang
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* These files are distributed at http://www.freesoftware.fsf.org/xlang/
 */

#include <xhash.h>
#include <stdio.h>
#include <stdlib.h>

#define HASH_TABLE_MIN_SIZE 11
#define HASH_TABLE_MAX_SIZE 13845163

static const unsigned int primes[] =
  {
    11,
    19,
    37,
    73,
    109,
    163,
    251,
    367,
    557,
    823,
    1237,
    1861,
    2777,
    4177,
    6247,
    9371,
    14057,
    21089,
    31627,
    47431,
    71143,
    106721,
    160073,
    240101,
    360163,
    540217,
    810343,
    1215497,
    1823231,
    2734867,
    4102283,
    6153409,
    9230113,
    13845163,
  };

static const unsigned int nprimes = sizeof (primes) / sizeof (primes[0]);

static void x_hash_table_resize (XHashTable *hash_table);
static XHashNode** x_hash_table_find_node (XHashTable *hash_table, void* key);
static XHashNode* x_hash_node_new (void* key, void* value);
static void x_hash_node_destroy (XHashNode *hash_node);
static void x_hash_nodes_destroy (XHashNode *hash_node);
static unsigned int primes_closest (unsigned int num);
static void x_hash_table_needresize(XHashTable *hash_table);

XHashTable* x_hash_table_new (HashFunc hash_func, EqualFunc key_equal_func)
{
  XHashTable *hash_table;
  unsigned int i;

  hash_table = x_new ( XHashTable );
  x_object_init_object (X_OBJECT (hash_table), x_hash_table_destroy);

  hash_table->size               = HASH_TABLE_MIN_SIZE;
  hash_table->nnodes             = 0;
  hash_table->hash_func          = hash_func ? hash_func : direct_hash;
  hash_table->key_equal_func     = key_equal_func;
  hash_table->nodes              = x_new_of_size ( XHashNode*, sizeof(XHashNode)*hash_table->size);

  for (i = 0; i < hash_table->size; i++)
    hash_table->nodes[i] = NULL;

  return hash_table;
}

void x_hash_table_destroy (XObject *hash_table)
{
  unsigned int i;
  XHashTable* real_table = (XHashTable*)hash_table;

  if ( real_table == NULL )return;

  for (i = 0; i < real_table->size; i++)
    x_hash_nodes_destroy (real_table->nodes[i]);

  x_destroy (real_table->nodes);
  x_destroy (real_table);
}

static XHashNode** x_hash_table_find_node (XHashTable *hash_table, void* key)
{
  XHashNode **node;

  node = &hash_table->nodes [(* hash_table->hash_func) (key) % hash_table->size];

  if (hash_table->key_equal_func)
    while (*node && !(*hash_table->key_equal_func) ((*node)->key, key))
      node = &(*node)->next;
  else
    while (*node && (*node)->key != key)
      node = &(*node)->next;

  return node;
}

void* x_hash_table_find (XHashTable *hash_table, void* key)
{
  XHashNode *node;
  if ( hash_table == NULL || key == NULL) return;

  node = *x_hash_table_find_node (hash_table, key);

  if (node)
    return x_addref(void*, node->value);
  else 
    return NULL;
}

void x_hash_table_insert (XHashTable *hash_table, void* key, void* value)
{
  XHashNode **node;

  if (hash_table == NULL)
    return;

  node = x_hash_table_find_node (hash_table, key);

  if (*node)
    {
      x_unref((*node)->value);
      (*node)->value = x_addref(void*, value);
    }
  else
    {
      *node = x_hash_node_new (key, value);
      hash_table->nnodes++;
      x_hash_table_needresize (hash_table);
    }
}

xbool x_hash_table_remove (XHashTable *hash_table, void*  key)
{
  XHashNode **node, *dest;

  if (hash_table == NULL) 
    return;

  node = x_hash_table_find_node (hash_table, key);
  if (*node)
    {
      dest = *node;
      (*node) = dest->next;
      x_hash_node_destroy (dest);
      hash_table->nnodes--;

      x_hash_table_needresize (hash_table);

      return TRUE;
    }

  return FALSE;
}

void x_hash_table_foreach(XHashTable *hash_table, UserFunc user_func, void* data)
{
  XHashNode *node;
  int i;
  
  if (hash_table == NULL || user_func == NULL)
    return;
  
  for (i = 0; i < hash_table->size; i++)
    for (node = hash_table->nodes[i]; node; node = node->next)
      (* user_func) (node->value, data);
}

unsigned int x_hash_table_get_size (XHashTable *hash_table)
{
  if (hash_table == NULL) 
    return;

  return hash_table->nnodes;
}

static void x_hash_table_needresize(XHashTable *hash_table)
{
  if ((hash_table->size >= 3*hash_table->nnodes && hash_table->size > HASH_TABLE_MIN_SIZE) ||
      (3 * hash_table->size <= hash_table->nnodes && hash_table->size < HASH_TABLE_MAX_SIZE))
    x_hash_table_resize (hash_table);
}

static void
x_hash_table_resize (XHashTable *hash_table)
{
  XHashNode **new_nodes;
  XHashNode *node;
  XHashNode *next;
  unsigned int hash_val;
  int new_size;
  int i;

  i = primes_closest(hash_table->nnodes);
  new_size = i > HASH_TABLE_MAX_SIZE ? HASH_TABLE_MAX_SIZE : i < HASH_TABLE_MIN_SIZE ? HASH_TABLE_MIN_SIZE : i ;

  new_nodes = x_new_of_size ( XHashNode*, sizeof(XHashNode)*new_size );

  for (i = 0; i < hash_table->size; i++)
    for (node = hash_table->nodes[i]; node; node = next)
      {
	next = node->next;
	
	hash_val = (* hash_table->hash_func) (node->key) % new_size;

	node->next = new_nodes[hash_val];
	new_nodes[hash_val] = node;
      }

  x_destroy (hash_table->nodes);
  hash_table->nodes = new_nodes;
  hash_table->size = new_size;
}

static XHashNode* x_hash_node_new (void* key, void* value)
{
  XHashNode *hash_node;

  hash_node = x_new ( XHashNode );

  hash_node->key = x_addref(void*, key);
  hash_node->value = x_addref(void*, value);;

  hash_node->next = NULL;

  return hash_node;
}

static void x_hash_node_destroy (XHashNode *hash_node)
{
  x_unref(hash_node->key);
  hash_node->key = NULL;

  x_unref(hash_node->value);
  hash_node->value = NULL;

  x_destroy(hash_node);
}

static void x_hash_nodes_destroy (XHashNode *hash_node)
{
  if (hash_node)
    {
      XHashNode *node = hash_node;
      XHashNode *temp;
      
      while (node->next)
	{
	  x_unref(node->key);
	  node->key = NULL;
	  
	  x_unref(node->value);
	  node->value = NULL;
	  
	  temp = node;
	  node = node->next;
	  x_destroy(temp);
	}
      
      x_unref(node->key);
      node->key = NULL;
      
      x_unref(node->value);
      node->value = NULL;
      x_destroy(node);
    }
}

static unsigned int primes_closest (unsigned int num)
{
  int i;

  for (i = 0; i < nprimes; i++)
    if (primes[i] > num)
      return primes[i];
  
  return primes[nprimes - 1];
}

unsigned int direct_hash (void* v)
{
  return (unsigned int)v; 
}
