/* X/lang - the eXtensible Language
 * Copyright (C) 2001 Patrick Deschenes
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* These files are distributed at http://xlang.sourceforge.net
 */

%{

#include <xltype.h>
#include <xlexpr.h>
#include <parser/basiclike/xllangbasic.h>
#include <xllang.h>

#ifndef YYDEBUG
#define	YYDEBUG	 0	/* Default to no yydebug support */
#endif

%}

%union {
  XLExpr*     expr;
  XLTypeType  xtype;
  xdouble     val_float;
  xint        val_int;
  xchar       val_string[100];
  xchar       identifier[100];
}

%token IF ELSE CLASS RETURN WHILE DYNAMIC STATIC V_TRUE V_FALSE AS END FUNCTION SUB THEN V_NULL
%token VOID CHAR UCHAR SHORT USHORT INT UINT FLOAT DOUBLE STRING BOOL HANDLE
%token<identifier> IDENTIFIER

%token<val_int>    V_INT
%token<val_float>  V_FLOAT
%token<val_string> V_STRING

%type<expr>  decl 
%type<expr>  decl_class decl_global decl_local decl_function
%type<expr>  class_prop class_method class_field
%type<expr>  code_line code_block code_block_or_line code_call code_assign code_return code_while code_if code_if_else
%type<expr>  type expr identifier literal expr_array
%type<xtype> type_basic;

%%

/*************
 * Start
 */

input: 
  /* Nothing */
  | input decl
;

decl:
    decl_class      { xl_lang_decl_add ($1); }
  | decl_global     { xl_lang_decl_add ($1); }
  | decl_function   { xl_lang_decl_add ($1); }
  | error           { xl_lang_basic_error ("Expecting a declaration"); } 
;

/****************
 * Declarations
 */

class_field_list: 
  /* Nothing */
  | class_field_list class_field { xl_lang_class_add ($2); }

class_field:   
    class_prop     { $$ = $1; }
  | class_method   { $$ = $1; }
  | error          { xl_lang_basic_error ("Expecting a class field (property or method"); }
;

class_prop: 
    type identifier ';'
    { $$ = xl_lang_efnct_create2 ("prop", $1, $2); }
  | type error ';' { xl_lang_basic_error ("Expecting an identifier\n"); }
;

class_method: 
  type identifier 
  { xl_lang_method_create ($1, $2);  }
  '(' decl_param_list ')'
  code_block
  { $$ = xl_lang_method_finish ($7); }
;

decl_class: 
    CLASS identifier
    { xl_lang_class_create ($2, NULL); }
    '{' class_field_list '}' 
    { $$ = xl_lang_class_finish (); }
  | CLASS error
    { xl_lang_basic_error ("Missing an identifier"); }
;

decl_global:
    identifier AS type
    { $$ = xl_lang_efnct_create2 ("global", $3, $1); }
  | identifier '(' ')' AS type '[' ']'
    {
      $1->type->array_type = XL_TYPE_ARRAY_TYPE_DYNAMIC;
      $1->type->array_size = 0;
      $$ = xl_lang_efnct_create2 ("global", $5, $1);
    }
  | identifier '[' V_INT ']' AS type
    {
      $6->type->array_type = XL_TYPE_ARRAY_TYPE_FIXED;
      $6->type->array_size = $3;
      $$ = xl_lang_efnct_create2 ("global", $6, $1);
    }
;

decl_local: 
    type identifier ';'  
    { $$ = xl_lang_efnct_create2 ("local", $1, $2); }
  | type identifier '[' ']' ';'
    {
      $1->type->array_type = XL_TYPE_ARRAY_TYPE_DYNAMIC;
      $1->type->array_size = 0;
      $$ = xl_lang_efnct_create2 ("local", $1, $2);
    }
  | type identifier '[' error ';'
    { xl_lang_basic_error ("Expecting a ']'\n"); }
  | type identifier '[' V_INT ']' ';'
    {
      $1->type->array_type = XL_TYPE_ARRAY_TYPE_FIXED;
      $1->type->array_size = $4;
      $$ = xl_lang_efnct_create2 ("local", $1, $2);
    }
  | type identifier '[' V_INT error ';'
    { xl_lang_basic_error ("Expecting a ']'\n"); }
;

decl_param:
    identifier AS type
    { xl_lang_fnct_add_param ($3, $1); }
  | identifier '[' ']' AS type
    {
      $5->type->array_type = XL_TYPE_ARRAY_TYPE_DYNAMIC;
      $5->type->array_size = 0;
      xl_lang_fnct_add_param ($5, $1);
    }
  | identifier '[' V_INT ']' AS type
    {
      $6->type->array_type = XL_TYPE_ARRAY_TYPE_FIXED;
      $6->type->array_size = $3;
      xl_lang_fnct_add_param ($6, $1);
    }
;

decl_param_list: 
  /* Nothing */
  | decl_param
  | decl_param_list ',' decl_param
;

decl_function: 
  FUNCTION identifier
  { xl_lang_fnct_create (NULL, $2);  }
  '(' decl_param_list ')'
  AS type
  code_block
  END FUNCTION
  { xl_lang_fnct_set_type ($8);
    $$ = xl_lang_fnct_finish ($9); 
  }
|
  SUB identifier
  { xl_lang_fnct_create (xl_lang_type_create (XL_TYPE_TYPE_VOID, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL), $2);  }
  '(' decl_param_list ')'
  code_block
  END SUB
  { $$ = xl_lang_fnct_finish ($7); }
;

/*****************
 * Basic elements
 */

type_basic:
    VOID      { $$ = XL_TYPE_TYPE_VOID;    }
  | CHAR      { $$ = XL_TYPE_TYPE_CHAR;    }
  | UCHAR     { $$ = XL_TYPE_TYPE_UCHAR;   }
  | SHORT     { $$ = XL_TYPE_TYPE_SHORT;   }
  | USHORT    { $$ = XL_TYPE_TYPE_USHORT;  }
  | INT       { $$ = XL_TYPE_TYPE_INT;     }
  | UINT      { $$ = XL_TYPE_TYPE_UINT;    }
  | STRING    { $$ = XL_TYPE_TYPE_STRING;  }
  | FLOAT     { $$ = XL_TYPE_TYPE_FLOAT;   }
  | DOUBLE    { $$ = XL_TYPE_TYPE_DOUBLE;  }
;

type:   
    type_basic                 { $$ = xl_lang_type_create ($1, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL); } 
  | DYNAMIC '<' identifier '>' { $$ = xl_expr_new (); }
  | STATIC '<' identifier '>'  { $$ = xl_expr_new (); } 
;

identifier:
  IDENTIFIER 
  { $$ =  xl_expr_new_identifier ($1); }
;

literal:   
    V_STRING   { $$ = xl_lang_data_string_create ($1);   }
  | V_INT      { $$ = xl_lang_data_integer_create ($1);  }
  | V_FLOAT    { $$ = xl_lang_data_float_create ($1);    }
  | V_TRUE     { $$ = xl_lang_data_bool_create (TRUE);   }
  | V_FALSE    { $$ = xl_lang_data_bool_create (FALSE);  }
;

/*******************
 * Expressions
 */

expr:   
    literal            { $$ = $1; }
  | code_call          { $$ = $1; }
  | expr_array         { $$ = $1; }
  | identifier         { $$ = $1; } 
  | expr '+' expr      { $$ = xl_lang_efnct_create2 ("+", $1, $3); }
  | expr '*' expr      { $$ = xl_lang_efnct_create2 ("*", $1, $3); }
  | expr '-' expr      { $$ = xl_lang_efnct_create2 ("-", $1, $3); }
  | expr '/' expr      { $$ = xl_lang_efnct_create2 ("/", $1, $3); }
  | expr '<' '=' expr  { $$ = xl_lang_efnct_create2 ("<=", $1, $4); }
  | expr '>' '=' expr  { $$ = xl_lang_efnct_create2 (">=", $1, $4); }
  | expr '<' expr      { $$ = xl_lang_efnct_create2 ("<", $1, $3); }
  | expr '>' expr      { $$ = xl_lang_efnct_create2 (">", $1, $3); }
  | expr '=' '=' expr  { $$ = xl_lang_efnct_create2 ("==", $1, $4); }
  | expr '<' '>' expr  { $$ = xl_lang_efnct_create2 ("<>", $1, $4); }
;

expr_array:
  expr '[' expr ']' { $$ = xl_lang_efnct_create2 ("[]", $1, $3); }
;

param_list: 
  /* Nothing */
  | expr                 { xl_lang_call_add ($1); }
  | param_list ',' expr  { xl_lang_call_add ($3); }
;

/*********************
 * Statements (CODE)
 */


code_lines: 
  /* Nothing */
  | code_lines code_line 
;

code_line:   
  /*    decl_local     { xl_lang_block_add ($1); } */
    code_call      { xl_lang_block_add ($1); }
  | code_assign    { xl_lang_block_add ($1); }
  | code_if        { xl_lang_block_add ($1); }
/*
  | code_return    { xl_lang_block_add ($1); }
  | code_while     { xl_lang_block_add ($1); }
*/
;

code_block: 
  { xl_lang_block_begin (); }  
  code_lines 
  { $$ = xl_lang_block_end (); }
;

code_block_or_line:
    code_block { $$ = $1; }
    | { xl_lang_block_begin (); } code_line { $$ = xl_lang_block_end (); }
;

code_call: 
  identifier
  { xl_lang_call_begin (x_string_get_str ($1->identifier)); }
  '(' param_list ')'
  { $$ = xl_lang_call_end (); }
;

code_assign: 
    expr '=' expr  { $$ = xl_lang_efnct_create2 ("=", $1, $3); }
;

code_return: 
    RETURN expr ';'   { $$ = xl_lang_efnct_create1 ("return", $2); }
  | RETURN error ';'
    { $$ = NULL;
      xl_lang_basic_error ("RETURN is expecting an <expr>");
      yyerrok; 
    }
  | RETURN expr error
    { $$ = NULL;
      xl_lang_basic_error ("Missing a semi-colon ';'");
      yyerrok;
    }
;

code_while:  
    WHILE '(' expr ')' code_block
    { $$ = xl_lang_efnct_create2 ("while", $3, $5); }
  | WHILE '(' error 
    { $$ = NULL;
      xl_lang_basic_error ("WHILE is expecting an <expr>");
      yyerrok; 
    }
  | WHILE error 
    { $$ = NULL;
      xl_lang_basic_error ("Missing an open parenthesis '('");
      yyerrok; 
    }
;

code_if_else:
  /* empty */ { $$ = NULL; }
  | ELSE code_block { $$ = $2; }
;

code_if:  
    IF 
    '(' expr ')' THEN
    code_block
    code_if_else
    END IF
    { $$ = xl_lang_efnct_create2 ("if", $3, $6); }
;

%%

