/* X Language - the eXtensible Language
 * Copyright (C) 2001 Patrick Deschenes
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* These files are distributed at http://www.freesoftware.fsf.org/xlang/
 */

%{

#include <xltype.h>
#include <xlexpr.h>
#include <parser/clike/xllangc.h>

#ifndef YYDEBUG
#define	YYDEBUG	 0	/* Default to no yydebug support */
#endif

%}

%union {
  XLExpr*     expr;
  XLTypeType  xtype;
  xdouble     vfloat;
  xint        vint;
  xchar       vchar;
  xchar       vstring[1000];
  xchar       identifier[1000];
}

%token DOMAIN CLASS INTERFACE USING ENUM
%token INHERIT CONST STATIC PUBLIC PROTECTED PRIVATE

%token VTRUE VFALSE VNULL

%token NEW 
%token IF ELSE RETURN FOR FOREACH IN DO WHILE SWITCH CASE DEFAULT BREAK CONTINUE TRY THROW CATCH
%token OP_SHR OP_SHL OP_INC OP_DEC OP_LOG_AND OP_LOG_OR

%token ARRAY_SIZE ARRAY_RESIZE ARRAY_ADD 
%token VOID CHAR UCHAR SHORT USHORT INT UINT FLOAT DOUBLE STRING BOOL

%token ASSIGN_SHR ASSIGN_SHL ASSIGN_ADD ASSIGN_SUB ASSIGN_MUL 
%token ASSIGN_DIV ASSIGN_MOD ASSIGN_AND ASSIGN_XOR ASSIGN_OR

%token OP_LE OP_GE OP_EQ OP_NE

%token<identifier> IDENTIFIER
%token<identifier> CLASSNAME


%token<vint>    VINT
%token<vchar>   VCHAR
%token<vfloat>  VFLOAT
%token<vstring> VSTRING

%token BOGUS

%type<expr> identifier
%type<expr> classname
%type<expr> literal

%type<expr>  type
%type<expr>  type_simple;
%type<xtype> type_element;

%type<expr> expr
%type<expr> expr_array
%type<expr> expr_access
%type<expr> expr_cast 

%type<expr>  decl 

%type<expr>  decl_domain decl_using

%type<expr>  decl_global

%type<expr>  decl_class
%type<expr>  class_field
%type<expr>  class_prop
%type<expr>  class_method

%type<expr> decl_fnct 
%type<expr> stmt 
%type<expr> stmt_block
%type<expr> stmt_block_or_line
%type<expr> stmt_local
%type<expr> stmt_call
%type<expr> stmt_invoke
%type<expr> stmt_assign
%type<expr> stmt_inc
%type<expr> stmt_dec
%type<expr> stmt_return
%type<expr> stmt_while
%type<expr> stmt_for
%type<expr> stmt_foreach
%type<expr> stmt_array_add
%type<expr> stmt_array_resize
%type<expr> stmt_if
%type<expr> stmt_else
%type<expr> stmt_new

%left '+' '-'
%left '*' '/'
%left '.'

%%

/*************
 * Start
 */

input: 
  /* Nothing */
  | input decl
;

decl:
    decl_domain            { xl_lang_decl_add ($1); }
  | decl_using             { xl_lang_decl_add ($1); }
  | decl_global            { xl_lang_decl_add ($1); }
  | decl_class             { xl_lang_decl_add ($1); }
  | decl_fnct              { xl_lang_decl_add ($1); } 
  | error                  { xl_lang_error ("not a declarator"); yyclearin; }
;

/* Domain
 */

decl_domain:
  DOMAIN identifier ';'
  { $$ = xl_lang_efnct_create1 ("@domain", $2); }
;

decl_using:
  USING identifier ';' { $$ = xl_lang_efnct_create1 ("@using", $2); }

/* Identifier
 */

identifier:
    IDENTIFIER                     { $$ =  xl_expr_new_identifier ($1); }
;

classname:
    CLASSNAME                      { $$ =  xl_expr_new_identifier ($1); }
;

/* Literal
 */

literal:   
    VSTRING        { $$ = xl_lang_data_string_create ($1);   }
  | VCHAR          { $$ = xl_lang_data_char_create ($1);  }
  | VINT           { $$ = xl_lang_data_integer_create ($1);  }
  | '-' VINT       { $$ = xl_lang_data_integer_create (-$2); } 
  | VFLOAT         { $$ = xl_lang_data_float_create ($1);    }
  | '-' VFLOAT     { $$ = xl_lang_data_float_create (-$2);   }
  | VTRUE          { $$ = xl_lang_data_bool_create (TRUE);   }
  | VFALSE         { $$ = xl_lang_data_bool_create (FALSE);  }
  | VNULL          { $$ = xl_lang_data_null_create (); }
;

/* Type
 */

type_element:
    VOID      { $$ = XL_TYPE_TYPE_VOID;    }
  | CHAR      { $$ = XL_TYPE_TYPE_CHAR;    }
  | UCHAR     { $$ = XL_TYPE_TYPE_UCHAR;   }
  | SHORT     { $$ = XL_TYPE_TYPE_SHORT;   }
  | USHORT    { $$ = XL_TYPE_TYPE_USHORT;  }
  | INT       { $$ = XL_TYPE_TYPE_INT;     }
  | UINT      { $$ = XL_TYPE_TYPE_UINT;    }
  | STRING    { $$ = XL_TYPE_TYPE_STRING;  }
  | FLOAT     { $$ = XL_TYPE_TYPE_FLOAT;   }
  | DOUBLE    { $$ = XL_TYPE_TYPE_DOUBLE;  }
  | BOOL      { $$ = XL_TYPE_TYPE_BOOL;    }
;

type_simple:
    type_element            { $$ = xl_lang_type_create ($1, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL); } 
  | classname               { $$ = xl_lang_type_create (XL_TYPE_TYPE_OBJECT, XL_TYPE_ARRAY_TYPE_NONE, 0, $1->identifier); 
                              x_unref ($1);
                            }
;

type:
    type_simple                 
    {
      $$ = $1; 
    }
  | type_simple '[' ']'         
    {
      $1->type->array_type = XL_TYPE_ARRAY_TYPE_DYNAMIC;
      $$ = $1;
    }
  | type_simple '[' VINT ']'
    {
      $1->type->array_type = XL_TYPE_ARRAY_TYPE_FIXED;
      $1->type->array_size = $3;
      $$ = $1; 
    }
;

/* Expressions
 */

expr:  
    literal               { $$ = $1; }
  | identifier            { $$ = $1; }
  | '(' expr ')'          { $$ = $2; }
  | expr_cast             { $$ = $1; } 

  | stmt_assign           { $$ = $1; }
  | stmt_call             { $$ = $1; }
  | stmt_inc              { $$ = $1; }
  | stmt_dec              { $$ = $1; }
  | stmt_invoke           { $$ = $1; }
  | expr_array            { $$ = $1; } 
  | expr_access           { $$ = $1; }  

  | expr '|' expr         { $$ = xl_lang_efnct_create2 ("|", $1, $3); }
  | expr '&' expr         { $$ = xl_lang_efnct_create2 ("&", $1, $3); }
  | expr '^' expr         { $$ = xl_lang_efnct_create2 ("^", $1, $3); }

  | expr '+' expr         { $$ = xl_lang_efnct_create2 ("+", $1, $3); }
  | expr '*' expr         { $$ = xl_lang_efnct_create2 ("*", $1, $3); }
  | expr '-' expr         { $$ = xl_lang_efnct_create2 ("-", $1, $3); }
  | expr '/' expr         { $$ = xl_lang_efnct_create2 ("/", $1, $3); }
  | expr '%' expr         { $$ = xl_lang_efnct_create2 ("%", $1, $3); }

  | '!' expr              { $$ = xl_lang_efnct_create1 ("!", $2); }      
  | expr OP_LE expr       { $$ = xl_lang_efnct_create2 ("<=", $1, $3); }
  | expr OP_GE expr       { $$ = xl_lang_efnct_create2 (">=", $1, $3); }
  | expr '<' expr         { $$ = xl_lang_efnct_create2 ("<", $1, $3); }
  | expr '>' expr         { $$ = xl_lang_efnct_create2 (">", $1, $3); }
  | expr OP_EQ expr       { $$ = xl_lang_efnct_create2 ("==", $1, $3); }
  | expr OP_NE expr       { $$ = xl_lang_efnct_create2 ("!=", $1, $3); }

  | '~' expr              { $$ = xl_lang_efnct_create1 ("~", $2); }      
  | expr OP_LOG_AND expr  { $$ = xl_lang_efnct_create2 ("&&", $1, $3); }
  | expr OP_LOG_OR expr   { $$ = xl_lang_efnct_create2 ("||", $1, $3); }
  | expr OP_SHL expr      { $$ = xl_lang_efnct_create2 ("<<", $1, $3); }
  | expr OP_SHR expr      { $$ = xl_lang_efnct_create2 (">>", $1, $3); }
;

expr_cast:
  '(' type ')' expr
   { $$ = xl_lang_efnct_create2 ("cast", $2, $4); }
;

expr_array:
    expr '[' expr ']'             { $$ = xl_lang_efnct_create2 ("[]", $1, $3); }
  | expr ARRAY_SIZE               { $$ = xl_lang_efnct_create1 ("@array_get_size", $1); }
;

expr_access:
  expr '.' identifier { $$ = xl_lang_efnct_create2 (".", $1, $3); }
;

expr_list: 
  /* Nothing */
  | expr                 { xl_lang_call_add ($1); }
  | expr_list ',' expr   { xl_lang_call_add ($3); }
;

/* Globals
 */

decl_global:
    type identifier ';'  
    { $$ = xl_lang_efnct_create2 ("@global", $1, $2); }
;

/* Classes
 */

class_field_list: 
    /* Nothing */
  | class_field_list class_field { xl_lang_class_add ($2); }
;

class_field:   
    class_prop     { $$ = $1; }
  | class_method   { $$ = $1; }
;

class_prop: 
    type identifier ';'
    { $$ = xl_lang_efnct_create2 ("@prop", $1, $2); }
;

class_method: 
    type identifier
    { xl_lang_method_create ($1, $2);  }
    '(' fnct_param_list ')'
    stmt_block
    { $$ = xl_lang_method_finish ($7); }
;

decl_class: 
    CLASS identifier
    { xl_lang_class_create ($2, NULL); }
    '{' 
    class_field_list 
    '}' 
    { $$ = xl_lang_class_finish (); }
;

/* Function
 */

fnct_param:
    type identifier 
    { xl_lang_fnct_add_param ($1, $2); }
;

fnct_param_list: 
  /* Nothing */
  | fnct_param
  | fnct_param_list ',' fnct_param
;

stmt_local: 
    type identifier
    { $$ = xl_lang_efnct_create2 ("@local", $1, $2); } 
;

stmts:
  /* Nothing */
  | stmts stmt
;

stmt:
    stmt_local ';'  { xl_lang_block_add ($1); }
  | stmt_block      { xl_lang_block_add ($1); }
  | stmt_call ';'   { xl_lang_block_add ($1); } 
  | stmt_invoke ';' { xl_lang_block_add ($1); }
  | stmt_assign ';' { xl_lang_block_add ($1); }
  | stmt_return     { xl_lang_block_add ($1); }
  | stmt_inc ';'    { xl_lang_block_add ($1); }
  | stmt_dec ';'    { xl_lang_block_add ($1); }
  | stmt_while      { xl_lang_block_add ($1); }
  | stmt_for        { xl_lang_block_add ($1); }
  | stmt_foreach    { xl_lang_block_add ($1); }
  | stmt_if         { xl_lang_block_add ($1); }
  | stmt_new        { xl_lang_block_add ($1); }
  | stmt_array_add ';'    { xl_lang_block_add ($1); }
  | stmt_array_resize ';' { xl_lang_block_add ($1); }
;

stmt_array_add:
  expr ARRAY_ADD '(' expr ')'   { $$ = xl_lang_efnct_create2 ("@array_add", $1, $4); }
;

stmt_array_resize:
  expr ARRAY_RESIZE '(' expr ')'   { $$ = xl_lang_efnct_create2 ("@array_set_size", $1, $4); }
;

stmt_block: 
  '{' 
  { xl_lang_block_begin (); }  
  stmts
  '}'
  { $$ = xl_lang_block_end (); }
;

stmt_block_or_line:
    stmt_block { $$ = $1; }
  | { xl_lang_block_begin (); } 
    stmt
    { $$ = xl_lang_block_end (); }
;

stmt_invoke: 
  expr '.' identifier
  { xl_lang_call_invoke_begin ($1, x_string_get_str ($3->identifier)); 
    x_unref ($3);
  }
  '(' expr_list ')'
  { $$ = xl_lang_call_end (); }
;

stmt_call: 
  identifier
  { xl_lang_call_begin (x_string_get_str ($1->identifier)); 
    x_unref ($1);
  }
  '(' expr_list ')'
  { $$ = xl_lang_call_end (); }
;

stmt_assign: 
    expr '=' expr  { $$ = xl_lang_efnct_create2 ("=", $1, $3); }
  | expr ASSIGN_SHR expr  { $$ = xl_lang_efnct_create2 (">>=", $1, $3); }
  | expr ASSIGN_SHL expr  { $$ = xl_lang_efnct_create2 ("<<=", $1, $3); }

  | expr ASSIGN_ADD expr  { $$ = xl_lang_efnct_create2 ("+=", $1, $3); }
  | expr ASSIGN_SUB expr  { $$ = xl_lang_efnct_create2 ("-=", $1, $3); }
  | expr ASSIGN_MUL expr  { $$ = xl_lang_efnct_create2 ("*=", $1, $3); }
  | expr ASSIGN_DIV expr  { $$ = xl_lang_efnct_create2 ("/=", $1, $3); }
  | expr ASSIGN_MOD expr  { $$ = xl_lang_efnct_create2 ("%=", $1, $3); }

  | expr ASSIGN_AND expr  { $$ = xl_lang_efnct_create2 ("&=", $1, $3); }
  | expr ASSIGN_OR expr  { $$ = xl_lang_efnct_create2 ("|=", $1, $3); }
  | expr ASSIGN_XOR expr  { $$ = xl_lang_efnct_create2 ("^=", $1, $3); }
;

stmt_inc: 
    expr OP_INC   { $$ = xl_lang_efnct_create1 ("++", $1); }

stmt_dec: 
    expr OP_DEC   { $$ = xl_lang_efnct_create1 ("--", $1); }

stmt_return: 
    RETURN ';'        { $$ = xl_lang_efnct_create ("@return"); }
  | RETURN expr ';'   { $$ = xl_lang_efnct_create1 ("@return", $2); }
;

stmt_for:
    FOR '(' expr ';' expr ';' expr ')' stmt_block_or_line
    { $$ = xl_lang_efnct_create4 ("@for", $3, $5, $7, $9); }
;

stmt_foreach:
    FOREACH '(' expr IN expr ')' stmt_block_or_line
    { $$ = xl_lang_efnct_create3 ("@foreach", $3, $5, $7); }
;

stmt_while:  
    WHILE '(' expr ')' stmt_block_or_line
    { $$ = xl_lang_efnct_create2 ("@while", $3, $5); }
;

stmt_new:
  expr '=' NEW type ';' { $$ = xl_lang_efnct_create2 ("@new", $1, $4); }
;

stmt_else:
  /* empty */ { $$ = NULL; }
  | ELSE stmt_block_or_line { $$ = $2; }
;

stmt_if:  
    IF '(' expr ')' 
    stmt_block_or_line
    stmt_else
    { if ($5)
        $$ = xl_lang_efnct_create2 ("@if", $3, $5); 
      else
        $$ = xl_lang_efnct_create1 ("@if", $3);

      if ($6)
	{
	  xl_expr_add ($$, $6);
	  x_unref ($6);
	}
    }
;

decl_fnct:
  type identifier
  { xl_lang_fnct_create ($1, $2);  }
  '(' fnct_param_list ')' 
  stmt_block
  { $$ = xl_lang_fnct_finish ($7); }
;

%%

