/* X Language - the eXtensible Language
 * Copyright (C) 2001 Patrick Deschenes
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* These files are distributed at http://www.freesoftware.fsf.org/xlang/
 */

#include <xldata.h>
#include <xlclass.h>
#include <xlfield.h>
#include <xlmain.h>

XLData*
xl_data_new (XLType* p_type, xptr p_ref)
{
  XLData*  self;
  
  self = x_new (XLData);
  x_object_init_object (X_OBJECT (self), xl_data_destroy);
  self->type = x_addref (XLType, p_type);
  xl_type_resolve (self->type);
  
  if (p_ref)
    {
      self->is_ref = TRUE;
      self->raw = p_ref;
      return self;
    }

  self->is_ref = FALSE;
  self->raw = NULL; 

  xl_data_init_data (self);

  return self;
}

XLData*
xl_data_new_simple (XLTypeType simple_type)
{
  XLData* self;
  XLType* type;

  type = xl_type_new (simple_type, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
  self = xl_data_new (type, NULL);
  x_unref (type);

  return self;
}

XLData*
xl_data_new_integer (xint value)
{
  XLData* self;

  self = xl_data_new_simple (XL_TYPE_TYPE_UINT);
  ((XLDataVariant*) self->raw)->vint = value;

  return self;
}

XLData*
xl_data_new_boolean (xbool value)
{
  XLData* self;

  self = xl_data_new_simple (XL_TYPE_TYPE_UINT);
  ((XLDataVariant*) self->raw)->vbool = vlaue;

  return self;
}

xvoid
xl_data_destroy (XObject* self)
{
  XString*         l_string;
  XLDataArray*     l_array;
  XLDataObject*    l_object;
  xuint            i, l_size;;

  if (!XL_DATA (self)->is_ref)
    {
      switch (XL_DATA (self)->type->array_type)
	{
	case XL_TYPE_ARRAY_TYPE_NONE:
	  
	  xl_datavariant_destroy ((XLDataVariant*) self->raw);
	  x_destroy (XL_DATA (self)->raw);
	  break;
	  
	case XL_TYPE_ARRAY_TYPE_FIXED:
	case XL_TYPE_ARRAY_TYPE_DYNAMIC:
	  l_array = ((XLDataArray*) XL_DATA (self)->raw);
	  if (l_array)
	    {
	      l_size = xl_data_array_get_size (XL_DATA (self));
	      x_destroy (l_array->data);
	      x_destroy (l_array);
	    }
	  break;
	}
    }

  x_unref (XL_DATA (self)->type);
  x_destroy (self);
}

xvoid
xl_data_init_data (XLData* self)
{
  XLClass*     l_class;
  XLData*      l_data;
  XLField*     l_field;
  XLType*      l_type;
  xuint	       i, l_size;    
  xptr         l_ptr;
  XLDataArray* l_array;
  XString*     l_string;

  switch (self->type->array_type)
    {
    case XL_TYPE_ARRAY_TYPE_NONE:

      switch (self->type->type)
	{
	case XL_TYPE_TYPE_STRING:
	  if (!self->is_ref)
	    self->raw = (xptr) x_string_new ();
	  break;

	case XL_TYPE_TYPE_BOOL:
	  if (!self->is_ref)
	    self->raw = x_alloc (sizeof (xbool));
	  x_access (self->raw, 0, xbool) = FALSE;
	  break;

	case XL_TYPE_TYPE_CHAR:
	case XL_TYPE_TYPE_UCHAR:
	  if (!self->is_ref)
	    self->raw = x_alloc (sizeof (xuchar));
	  x_access (self->raw, 0, xuchar) = 0;
	  break;

	case XL_TYPE_TYPE_SHORT:
	case XL_TYPE_TYPE_USHORT:
	  if (!self->is_ref)
	    self->raw = x_alloc (sizeof (xushort));
	  x_access (self->raw, 0, xushort) = 0;
	  break;
      
	case XL_TYPE_TYPE_INT:
	case XL_TYPE_TYPE_UINT:
	  if (!self->is_ref)
	    self->raw = x_alloc (sizeof (xuint));	  
	  x_access (self->raw, 0, xuint) = 0;
	  break;

	case XL_TYPE_TYPE_FLOAT:
	  if (!self->is_ref)
	    self->raw = x_alloc (sizeof (xfloat));	  
	  x_access (self->raw, 0, xfloat) = 0.0;
	  break;

	case XL_TYPE_TYPE_DOUBLE:
	  if (!self->is_ref)
	    self->raw = x_alloc (sizeof (xdouble));
	  x_access (self->raw, 0, xdouble) = 0.0;
	  break;
    
	case XL_TYPE_TYPE_OBJECT:
	  l_class = xl_type_get_class (self->type);
	  if (l_class)
	    {
	      if (!self->is_ref)
		self->raw = x_alloc (sizeof (XLDataObject));
	    }
	  break;
	}
      break;

    case XL_TYPE_ARRAY_TYPE_FIXED:
      if (!self->is_ref)
	self->raw = x_alloc (sizeof (XLDataArray));

      l_array = (XLDataArray*) self->raw;

      l_array->size = self->type->array_size;

      l_type = xl_type_new (self->type->type, XL_TYPE_ARRAY_TYPE_NONE, 0, self->type->classname);
      l_array->data = x_alloc (l_array->size * xl_type_get_size (l_type));
      x_unref (l_type);

      for (i=0; i<l_array->size; i++)
	{
	  l_data = xl_data_get_index (self, i);
	  xl_data_init_data (l_data);
	  x_unref (l_data);
	}

      break;

    case XL_TYPE_ARRAY_TYPE_DYNAMIC:
      if (!self->is_ref)
	self->raw = x_alloc (sizeof (XLDataArray));

      l_array = (XLDataArray*) self->raw;

      l_array->size = self->type->array_size;
      l_array->data = x_alloc (l_array->size * sizeof (xl_type_get_size (self->type)));
      break;
    }
}

xvoid
xl_data_destroy_data (XLData* self)
{
}

xvoid
xl_data_set_type (XLData* self, XLType* p_type)
{
}

XString*
xl_data_get_string (XLData* self)
{
  return x_addref (XString, self->raw);
}

xvoid
xl_data_set_string (XLData* self, XString* p_value)
{
  x_string_set ((XString*) self->raw, p_value);
}

xptr
xl_data_get_raw (XLData* self)
{
  return self->raw;
}

xvoid
xl_data_assign (XLData* self, XLData* p_src)
{
  XLDataObject*  l_objsrc;
  XLDataObject*  l_objdst;
  XString*       l_text;
  XLClass*       l_class;
  xchar          l_str[1000];

  if (!p_src)
    {
      printf ("ASSIGN ERROR: Source not available\n");
      return;
    }

  if (self->type->type == XL_TYPE_TYPE_STRING)
    {
      if (p_src->type->type == XL_TYPE_TYPE_STRING)
	{
	  l_text = xl_data_get_string (p_src);
	  xl_data_set_string (self, l_text);
	  x_unref (l_text);
	}
      if (p_src->type->type == XL_TYPE_TYPE_UINT)
	{
	  sprintf (l_str, "%d", x_access (p_src->raw, 0, xuint));
	  l_text = x_string_new ();
	  x_string_set_str (l_text, l_str);
	  xl_data_set_string (self, l_text);
	  x_unref (l_text);
	}
      if (p_src->type->type == XL_TYPE_TYPE_FLOAT)
	{
	  sprintf (l_str, "%f", x_access (p_src->raw, 0, xfloat));
	  l_text = x_string_new ();
	  x_string_set_str (l_text, l_str);
	  xl_data_set_string (self, l_text);
	  x_unref (l_text);
	}
      if (p_src->type->type == XL_TYPE_TYPE_DOUBLE)
	{
	  sprintf (l_str, "%f", x_access (p_src->raw, 0, xdouble));
	  l_text = x_string_new ();
	  x_string_set_str (l_text, l_str);
	  xl_data_set_string (self, l_text);
	  x_unref (l_text);
	}
    }

  if (self->type->type == XL_TYPE_TYPE_BOOL)
    x_access (self->raw, 0, xbool) = x_access (p_src->raw, 0, xbool);

  if (self->type->type == XL_TYPE_TYPE_CHAR)
    x_access (self->raw, 0, xchar) = x_access (p_src->raw, 0, xchar);

  if (self->type->type == XL_TYPE_TYPE_UCHAR)
    x_access (self->raw, 0, xuchar) = x_access (p_src->raw, 0, xuchar);

  if (self->type->type == XL_TYPE_TYPE_SHORT)
    x_access (self->raw, 0, xshort) = x_access (p_src->raw, 0, xshort);

  if (self->type->type == XL_TYPE_TYPE_USHORT)
    x_access (self->raw, 0, xushort) = x_access (p_src->raw, 0, xushort);

  if (self->type->type == XL_TYPE_TYPE_INT)
    x_access (self->raw, 0, xint) = x_access (p_src->raw, 0, xint);

  if (self->type->type == XL_TYPE_TYPE_UINT)
    x_access (self->raw, 0, xuint) = x_access (p_src->raw, 0, xuint);

  if (self->type->type == XL_TYPE_TYPE_FLOAT)
    x_access (self->raw, 0, xfloat) = x_access (p_src->raw, 0, xfloat);

  if (self->type->type == XL_TYPE_TYPE_DOUBLE)
    x_access (self->raw, 0, xdouble) = x_access (p_src->raw, 0, xdouble);

  if (self->type->type == XL_TYPE_TYPE_OBJECT)
    {
      l_objdst = ((XLDataObject*) self->raw);
      l_objsrc = ((XLDataObject*) p_src->raw);

      l_objdst->refcount--;
      l_objdst->refcount++;
    }
}

XLData*
xl_data_copy (XLData* self)
{
  XLData* l_data;

  l_data = xl_data_new (self->type, NULL);
  xl_data_assign (l_data, self);

  return l_data;
}

XLData*
xl_data_addref_field (XLData* self, XLField* p_field)
{
  return NULL;
}

XLData*
xl_data_convert (XLData* self, XLType* p_type)
{
  return NULL;
}

XLData*
xl_data_get_index (XLData* self, xuint p_index)
{
  XLDataArray* l_array;
  XLData*      l_data;
  XLType*      l_type;
  xptr         l_raw;
  xuint        l_size;

  switch (self->type->array_type)
    {
    case XL_TYPE_ARRAY_TYPE_FIXED:

      l_type = xl_type_new (self->type->type, XL_TYPE_ARRAY_TYPE_NONE, 0, self->type->classname);
      l_size = xl_type_get_size (l_type);

      l_array = (XLDataArray*) self->raw;
      l_raw = l_array->data;
      l_data = xl_data_new (l_type, &x_access (l_raw, (p_index*l_size), xptr));
      x_unref (l_type);
      break;

    case XL_TYPE_ARRAY_TYPE_DYNAMIC:

      l_type = xl_type_new (self->type->type, XL_TYPE_ARRAY_TYPE_NONE, 0, self->type->classname);
      l_size = xl_type_get_size (l_type);

      l_array = (XLDataArray*) self->raw;
      l_raw = l_array->data;

      if (p_index >= 0 && p_index < l_array->size)
	l_data = xl_data_new (l_type, &x_access (l_raw, (p_index*l_size), xptr));
      else
	{
	  printf ("error: array outbound (%d, %d)\n", l_array->size, p_index);
	  exit (-1);
	}

      x_unref (l_type);
      break;

    default:

      l_data = NULL;
      break;
    }

  return l_data;
}

xuint
xl_data_array_get_size (XLData* self)
{
  x_assert (self != NULL);

  switch (self->type->array_type)
    {
    case XL_TYPE_ARRAY_TYPE_FIXED:
      return self->type->array_size;
    case XL_TYPE_ARRAY_TYPE_DYNAMIC:
      return ((XLDataArray*) self->raw)->size;
    default:
      return 0;
    }
}

xvoid
xl_data_array_set_size (XLData* self, xuint p_newsize)
{
  XLDataArray* l_array;
  XLData*      l_data;
  xuint        l_oldsize;
  xuint        i;

  x_assert (self != NULL);

  switch (self->type->array_type)
    {
    case XL_TYPE_ARRAY_TYPE_FIXED:
      printf ("error: unable to resize a fixed array\n");
      break;
    case XL_TYPE_ARRAY_TYPE_DYNAMIC:
      l_array = (XLDataArray*) self->raw;
      l_oldsize = l_array->size;
      l_array->size = p_newsize;
      l_array->data = x_resize (l_array->data, l_array->size * xl_type_get_size (self->type));

      /* Initialize the expanded array
       */

      if (p_newsize > l_oldsize)
	{
	  for (i=l_oldsize; i<p_newsize; i++)
	    {
	      l_data = xl_data_get_index (self, i);
	      xl_data_init_data (l_data);
	      x_unref (l_data);
	    }
	}

      break;
    }
}

XLData*
xl_data_get_field (XLData* self, XString* p_fieldname, xbool p_force)
 {
  XLClass*      l_class;
  XLType*       l_type;
  XLData*       l_data;
  XLField*      l_field;
  xptr          l_raw;
  XLDataObject* l_obj;

  if (self->type->type == XL_TYPE_TYPE_OBJECT)
    {
      l_obj = ((XLDataObject*) self->raw);
      if (!l_obj)
	{
	  printf ("error: null object\n");
	  exit (-1);
	}
      l_class = x_addref (XLClass, l_obj->class);
    }
  else
    l_class = x_addref (XLClass, xl_type_get_class (self->type));

  l_field = xl_class_get_field (l_class, p_fieldname);
  if (!l_field)
    {
      if (p_force)
	{
	  printf ("error: field '%s' in class '%s' not available\n", 
		  x_string_get_str (p_fieldname), x_string_get_str (l_class->name));
	  exit (-1);
	}
      else
	return NULL;
    }

  switch (self->type->type)
    {
    case XL_TYPE_TYPE_OBJECT:
      l_obj = ((XLDataObject*) self->raw);
      if (!l_obj)
	{
	  printf ("warning: uninitialized object\n");
	  return NULL;
	}
      l_data = xl_data_new (l_field->type, ((xuchar*) l_raw) + l_field->index);
      break;
    default:
      l_data = NULL;
      break;
    }

  x_unref (l_class);
  x_unref (l_field);

  return l_data;
}

xptr
xl_data_get_ptr (XLData* self)
{
  return NULL;
}

xvoid
xl_data_set_raw (XLData* self, xptr p_raw)
{
  if (self->type->array_type != XL_TYPE_ARRAY_TYPE_NONE)
    {
      self->raw = p_raw;
    }
}

XLClass*
xl_data_get_class (XLData* self)
{
  XLDataObject* l_objref;

  l_objref = ((XLDataObject*) self->raw);
  if (l_objref)
    return x_addref (XLClass, l_objref->class);
  else
    return NULL;
}

xvoid
xl_data_object_new (XLData* self, XLClass* p_class)
{
  XLDataObject*   l_objref;
  XLField*        l_field;
  XLData*         l_data;
  XLClass*        l_class;
  XString*        l_name;
  XLExpr*         l_invoke;
  XLExpr*         l_self;
  XLFnct*         l_fnct;
  xint            i;
  xuint           l_size;

  xl_class_init_class (p_class);

  l_objref = (XLDataObject*) self->raw;
  l_objref = x_alloc (sizeof (XLDataObject));
  l_objref->refcount = 1;
  l_objref->class = x_addref (XLClass, p_class);
  
  /* We loop in all fields
   */
  
  l_class = p_class;

  while (l_class)
    {
      l_size = x_list_get_size (l_class->list_fields);
      for (i = 0; i < l_size; i++)
	{
	  l_field = XL_FIELD (x_list_get (l_class->list_fields, i));
	  l_data = xl_data_get_field (self, l_field->name, TRUE);
	  xl_data_init_data (l_data);
	  
	  x_unref (l_data);
	  x_unref (l_field);
	}

      l_class = l_class->parent_class;
    }

  /* Call the constructor
   */

  l_class = p_class;
  l_name = x_string_new ();
  x_string_set (l_name, l_class->name);
  x_string_add_str (l_name, "::");
  x_string_add (l_name, l_class->name);

  l_fnct = xl_main_get_fnct (g_xl_main_global, l_name);

  if (l_fnct)
    {
      l_invoke = xl_expr_new ();
      xl_expr_set_fnct (l_invoke, l_name);
      
      l_self = xl_expr_new ();
      xl_expr_set_data (l_self, self);
      
      xl_expr_add (l_invoke, l_self);
      x_unref (l_self);
      
      xl_expr_evaluate (l_invoke);

      x_unref (l_fnct);
    }

  x_unref (l_name);
}

xbool
xl_data_read_integer (XLData* self, xint* value)
{
  if (self->type->array_type == XL_TYPE_ARRAY_TYPE_NONE)
    {
      if (self->type->type == XL_TYPE_TYPE_SHORT || self->type->type == XL_TYPE_TYPE_USHORT ||
	  self->type->type == XL_TYPE_TYPE_INT || self->type->type == XL_TYPE_TYPE_UINT)
	{
	  *value = x_access (self->raw, 0, xint);
	  return TRUE;
	}
      else
	return FALSE;
    }
  else
    return FALSE;
}
