/* XLanguage - the eXtensible Language
 * Copyright (C) 2001 Patrick Deschenes
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* These files are distributed at http://www.freesoftware.fsf.org/xlang/
 */

#include <xlexpr.h>
#include <xlmain.h>

typedef struct _XLPkgHeader XLPkgHeader;
struct _XLPkgHeader
{
  xuchar  magic[4];
  xuchar  extra[12];
};

XLExpr*
xl_expr_new ()
{
  XLExpr *self;

  self = x_new (XLExpr);
  x_object_init_object (X_OBJECT (self), xl_expr_destroy);
  
  self->expr_type = XL_EXPR_TYPE_NONE;
  self->data = NULL;
  self->fnct = NULL;
  self->type = NULL;
  self->identifier = NULL;
  self->source_file = NULL;
  self->source_line = 0;
  self->check = TRUE;
  
  self->list_expr = x_list_new ();
 
  return self;
}

XLExpr*
xl_expr_new_fnct (xstr l_fnct)
{
  XLExpr*	self;
  XString*	l_name;
  
  self = xl_expr_new ();
  l_name = x_string_new ();
  x_string_set_str (l_name, l_fnct);
  xl_expr_set_fnct (self, l_name);
  x_unref (l_name);
  
  return self;
}

XLExpr*
xl_expr_new_identifier (xstr p_identifier)
{
  XLExpr*	self;
  XString*	l_name;
  
  self = xl_expr_new ();
  l_name = x_string_new ();
  x_string_set_str (l_name, p_identifier);
  xl_expr_set_identifier (self, l_name);
  x_unref (l_name);
  
  return self;
}

xvoid
xl_expr_destroy (XObject* self)
{
  x_unref (XL_EXPR (self)->data);
  x_unref (XL_EXPR (self)->fnct);
  x_unref (XL_EXPR (self)->type);
  x_unref (XL_EXPR (self)->source_file);
  x_unref (XL_EXPR (self)->identifier);
  x_unref (XL_EXPR (self)->list_expr);

  x_destroy (self);
}

xvoid
xl_expr_add (XLExpr* self, XLExpr* p_expr)
{
  x_list_add (self->list_expr, X_OBJECT (p_expr));
}

xvoid
xl_expr_evaluate (XLExpr* self)
{
  XString* l_name;
  XLExpr*  l_subexpr;
  XLFnct*  l_fnct;
  XLData*  l_data;
  XLVar*   l_var;
  xuint    i, l_size;

  if (!self) return;
  if (g_xl_fnct_halt) return;

  if (g_xl_main_global->debugger)
    xl_debugger_break (self);

  switch (self->expr_type)
    {
    case XL_EXPR_TYPE_NONE:
      break;
    case XL_EXPR_TYPE_BLOCK:
      l_size = x_list_get_size (self->list_expr);
      for (i = 0; i < l_size; i++)
	{
	  l_subexpr = XL_EXPR (x_list_get (self->list_expr, i));
	  xl_expr_evaluate (l_subexpr);
	  x_unref (l_subexpr);
	}
      break;
    case XL_EXPR_TYPE_DATA:
       break;
    case XL_EXPR_TYPE_TYPE:
      xl_type_resolve (self->type);
      break;
    case XL_EXPR_TYPE_IDENTIFIER:
      l_name = x_string_new ();
      x_string_set_str (l_name, "self");

      /* If 'self' is defined, we look for an 'attribute' before
       * looking for a variable.
       */

      if (x_string_cmp_str (self->identifier, "self") != 0)
	{
	  l_var = xl_main_get_var (g_xl_main_global, l_name);
	  if (l_var)
	    {
	      l_data = xl_data_get_field (l_var->data, self->identifier, FALSE);
	      x_unref (l_var);
	      if (l_data)
		{
		  x_unref (l_name);
		  x_unref (self->data);
		  self->data = l_data;
		  return;
		}
	    }
	}

      x_unref (l_name);

      l_var = xl_main_get_var (g_xl_main_global, self->identifier);
      if (l_var)
	{
	  x_unref (self->data);
	  self->data = x_addref (XLData, l_var->data);
	  x_unref (l_var);
	}
      else
	{
	  x_unref (self->data);
	  self->data = NULL;

	  l_fnct = xl_main_get_fnct (g_xl_main_global, self->identifier);
	  if (l_fnct)
	    {
	      XLType* l_type;
	      XLData* l_data;
	      
	      l_type = x_addref (XLType, l_fnct->ret_type);
	      l_data = xl_data_new (l_type, NULL);

	      x_unref (x_access (xl_data_get_raw (l_data), 0, XLFnct*));
	      x_access (xl_data_get_raw (l_data), 0, XLFnct*) = x_addref (XLFnct, l_fnct); 

	      x_unref (l_type);
	      x_unref (l_fnct);

	      x_unref (self->data);
	      self->data = l_data;
	    }
	  else
	    {
	      if (self->check)
		{
		  printf ("ERROR: variable '%s' not founded\n", x_string_get_str (self->identifier));
		  exit (0);
		}
	    }
	}
      break;
    case XL_EXPR_TYPE_FNCT:
      
      if (!(self->fnct))
	self->fnct = xl_main_get_fnct (g_xl_main_global, self->identifier);

      if (self->fnct)
	{
	  x_unref (self->data);
	  self->data = NULL;

	  l_data = xl_fnct_call (self->fnct, self);
	  if (l_data)
	    self->data = l_data;
	}
      else
	{
	  printf ("ERROR: function '%s' not founded\n", x_string_get_str (self->identifier));
	  exit (-1);
	}

      break;
    default:
      break;
    }
}

xbool
xl_expr_check (XLExpr* self, XLExprType p_type)
{
  if (self->expr_type == p_type)
    return TRUE;
  else
    return FALSE;
}

XLExpr*
xl_expr_get (XLExpr* self, xuint p_index)
{
  return XL_EXPR (x_list_get (self->list_expr, p_index));
}

xuint
xl_expr_get_size (XLExpr* self)
{
  return x_list_get_size (self->list_expr);
}

xvoid
xl_expr_set_block (XLExpr* self)
{
  self->expr_type = XL_EXPR_TYPE_BLOCK;
}

xvoid
xl_expr_set_data (XLExpr* self, XLData* p_data)
{
  self->expr_type = XL_EXPR_TYPE_DATA;

  x_unref (self->data);
  self->data = x_addref (XLData, p_data);
}

xvoid
xl_expr_set_fnct (XLExpr* self, XString* p_name)
{
  self->expr_type = XL_EXPR_TYPE_FNCT;

  x_unref (self->identifier);
  self->identifier = x_addref (XString, p_name);
}

xvoid
xl_expr_set_type (XLExpr* self, XLType* p_type)
{
  self->expr_type = XL_EXPR_TYPE_TYPE;

  x_unref (self->type);
  self->type = x_addref (XLType, p_type);
}

xvoid
xl_expr_set_identifier (XLExpr* self, XString* p_identifier)
{
  self->expr_type = XL_EXPR_TYPE_IDENTIFIER;

  x_unref (self->identifier);
  self->identifier = x_addref (XString, p_identifier);
}

xbool
xl_expr_read_integer (XLExpr* self, xuint index, xint* value)
{
  XLExpr* subexpr;
  
  subexpr = XL_EXPR (x_list_get (self->list_expr, index));
  xl_expr_evaluate (subexpr);

  if (xl_data_read_integer (subexpr->data, value))
    {
      x_unref (subexpr);
      return TRUE;
    }
  else
    {
      x_unref (subexpr);
      return FALSE;
    }
}

xuint nb_tab = 0;

xvoid
xl_expr_debug (XLExpr* self)
{
  xuint i;
  xuint l_size;

  if (!self)
    return;

  if (self->expr_type == XL_EXPR_TYPE_FNCT)
    {
      for (i=0; i<nb_tab; i++)
	printf (" ");
      
      printf ("FNCT '%s'\n", x_string_get_str (self->identifier));
      nb_tab++;

      l_size = x_list_get_size (self->list_expr);      
      for (i=0; i<l_size; i++)
	xl_expr_debug (xl_expr_get (self, i));

      nb_tab--;
    }

  if (self->expr_type == XL_EXPR_TYPE_BLOCK)
    {
      for (i=0; i<nb_tab; i++)
	printf (" ");
      
      printf ("BLOCK\n");
      nb_tab++;

      l_size = x_list_get_size (self->list_expr);      
      for (i=0; i<l_size; i++)
	xl_expr_debug (xl_expr_get (self, i));

      nb_tab--;
    }
}

xvoid
xl_expr_load_chunk (XLExpr* self, FILE* file)
{
  XString* l_string;
  XLExpr*  l_expr;
  XLData*  l_data;
  XLType*  l_type;
  xuchar   type;
  xchar    ch;
  xuchar   nativetype;
  xuchar   arraytype;
  xuint    arraysize;
  xuint    childs;
  xuint    size;
  xuint    i;

  l_expr = xl_expr_new ();

  fread (&type, sizeof (type), 1, file);
  fread (&childs, sizeof (childs), 1, file);
  
  switch ((XLExprType) type)
    {
    case XL_EXPR_TYPE_NONE:
      break;
    case XL_EXPR_TYPE_BLOCK:
      xl_expr_set_block (l_expr);
      break;
    case XL_EXPR_TYPE_DATA:
      fread (&nativetype, sizeof (nativetype), 1, file);
      switch ((XLTypeType) nativetype)
	{
	case XL_TYPE_TYPE_STRING:
	  fread (&size, sizeof (size), 1, file);
	  l_string = x_string_new ();
	  for (i=0; i<size; i++)
	    {
	      fread (&ch, sizeof (ch), 1, file);
	      x_string_add_char (l_string, ch);
	    }

	  l_type = xl_type_new ((XLTypeType) nativetype, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
	  l_data = xl_data_new (l_type, NULL);
	  xl_data_set_string (l_data, l_string);
	  xl_expr_set_data (l_expr, l_data);

	  x_unref (l_data);
	  x_unref (l_type);
	  x_unref (l_string);

	  break;
	default:
	  l_type = xl_type_new ((XLTypeType) nativetype, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
	  l_data = xl_data_new (l_type, NULL);

	  fread (l_data->raw, xl_type_get_size (l_type), 1, file);
	  xl_expr_set_data (l_expr, l_data);

	  x_unref (l_data);
	  x_unref (l_type);

	  break;
	}
      break;
    case XL_EXPR_TYPE_FNCT:
      fread (&size, sizeof (size), 1, file);

      l_string = x_string_new ();
      for (i=0; i<size; i++)
	{
	  fread (&ch, sizeof (ch), 1, file);
	  x_string_add_char (l_string, ch);
	}

      xl_expr_set_fnct (l_expr, l_string);
      x_unref (l_string);
      break;
    case XL_EXPR_TYPE_TYPE:
      
      fread (&nativetype, sizeof (nativetype), 1, file);
      fread (&arraytype, sizeof (arraytype), 1, file);
      fread (&arraysize, sizeof (arraysize), 1, file);

      switch ((XLTypeType) nativetype)
	{
	default:
	  
	  l_type = xl_type_new ((XLTypeType) nativetype, (XLTypeArrayType) arraytype, arraysize, NULL);
	  xl_expr_set_type (l_expr, l_type);
	  x_unref (l_type);
	  break;
	}
     
      break;
    case XL_EXPR_TYPE_IDENTIFIER:
      fread (&size, sizeof (size), 1, file);

      l_string = x_string_new ();
      for (i=0; i<size; i++)
	{
	  fread (&ch, sizeof (ch), 1, file);
	  x_string_add_char (l_string, ch);
	}

      xl_expr_set_identifier (l_expr, l_string);
      x_unref (l_string);     
      break;
    default:
      printf ("error: unknown type\n");
      break;
    }

  for (i=0; i<childs; i++)
    {
      xl_expr_load_chunk (l_expr, file);
    }

  xl_expr_add (self, l_expr);
  x_unref (l_expr);
}

xvoid
xl_expr_load (XLExpr* self, XString* p_filename)
{
  FILE*       file;
  XLPkgHeader header;

  file = fopen (x_string_get_str (p_filename), "r");

  /* : size_t fread (void *data, size_t size, size_t count, FILE *stream)  */

  /* Read header
   */

  fread (&header, sizeof (header), 1, file);
  if (header.magic[0] == 'X' && header.magic[1] == 'P' &&
      header.magic[2] == 'K' && header.magic[3] == 'G')
    {
      xl_expr_load_chunk (self, file);
    }

  fclose (file);
}

xvoid
xl_expr_save_chunk (XLExpr* self, FILE* file)
{
  XString* l_string;
  XLExpr*  l_expr;
  xuchar   type;
  xuchar   nativetype;
  xuchar   arraytype;
  xuint    arraysize;
  xuint    childs;
  xuint    size;
  xuint    i;

  type = (xuchar) self->expr_type;
  childs = x_list_get_size (self->list_expr);

  fwrite (&type, sizeof (type), 1, file);
  fwrite (&childs, sizeof (childs), 1, file);
  
  switch (self->expr_type)
    {
    case XL_EXPR_TYPE_NONE:
      break;
    case XL_EXPR_TYPE_BLOCK:
      break;
    case XL_EXPR_TYPE_DATA:
      nativetype = self->data->type->type;

      fwrite (&nativetype, sizeof (nativetype), 1, file);
      switch ((XLTypeType) nativetype)
	{
	case XL_TYPE_TYPE_STRING:
	  l_string = xl_data_get_string (self->data);

	  size = x_string_get_size (l_string);
	  fwrite (&size, sizeof (size), 1, file);
	  fwrite (x_string_get_str (l_string), size, 1, file);

	  x_unref (l_string);
	  break;
	default:
	  fwrite (self->data->raw, xl_type_get_size (self->data->type), 1, file);
	  break;
	}
      break;
    case XL_EXPR_TYPE_FNCT:
      size = x_string_get_size (self->identifier);
      fwrite (&size, sizeof (size), 1, file);
      fwrite (x_string_get_str (self->identifier), size, 1, file);
      break;
    case XL_EXPR_TYPE_TYPE:
      nativetype = self->type->type;
      arraytype = self->type->array_type;
      arraysize = self->type->array_size;

      fwrite (&nativetype, sizeof (nativetype), 1, file);
      fwrite (&arraytype, sizeof (arraytype), 1, file);
      fwrite (&arraysize, sizeof (arraysize), 1, file);
     
      break;
    case XL_EXPR_TYPE_IDENTIFIER:
      size = x_string_get_size (self->identifier);
      fwrite (&size, sizeof (size), 1, file);
      fwrite (x_string_get_str (self->identifier), size, 1, file);
      break;
    default:
      printf ("error: unknown type\n");
      break;
    }

  for (i=0; i<childs; i++)
    {
      l_expr = xl_expr_get (self, i);
      xl_expr_save_chunk (l_expr, file);
      x_unref (l_expr);
    }
}

xvoid
xl_expr_save (XLExpr* self, XString* p_filename)
{
  XLPkgHeader header;
  FILE*       file;

  file = fopen (x_string_get_str (p_filename), "w+");

  /* Write header
   */

  memset (&header, 0, sizeof (header));
  header.magic[0] = 'X';
  header.magic[1] = 'P';
  header.magic[2] = 'K';
  header.magic[3] = 'G';
  fwrite (&header, sizeof (header), 1, file);

  xl_expr_save_chunk (self, file);

  fclose (file);
}
