/* 
 * Yogi (GUI for clens)
 * yogi.c 
 *
 * Copyright (C) 2006 Jean Bréfort <jean.brefort@normalesup.org>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301
 * USA
 */

#include "config.h"
#include <glade/glade.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <gconf/gconf-client.h>
#include <string.h>

GtkImage *source, *dest;
GtkLabel *camera_lbl;
GtkComboBox *lenses_box;
GtkListStore *lenses_list;
GtkTreeSelection *lenses_sel;
GtkSpinButton *jpeg_btn;
GtkEntry *ext_entry;
GtkToggleButton *tiff_btn, *unscale_btn;
GtkFileChooser *profile_sel, *stitcher_sel;
GtkWidget *prefs;
int width = 240, height = 180;
char const *filename = NULL;
char *profile_path = NULL;
char *stitcher = NULL;
GConfClient *conf_client;
unsigned notification_id;
GdkCursor *cursor;
GdkWindow *main_win;

void config_changed_cb (GConfClient *client,  guint cnxn_id, GConfEntry *entry, gpointer user_data)
{
	g_return_if_fail (client == conf_client);
	g_return_if_fail (cnxn_id == notification_id);
	if (!strcmp (entry->key, "/apps/yogi/profile_path")) {
		g_free (profile_path);
		profile_path = g_strdup (gconf_value_get_string (entry->value));
	}
	else if (!strcmp (entry->key, "/apps/yogi/stitcher")) {
		g_free (stitcher);
		stitcher = g_strdup (gconf_value_get_string (entry->value));
	}
}

void about_cb ()
{
	char * authors[] = {"Jean Bréfort", "Members of COAGUL http://www.coagul.org", NULL};
	char * artists[] = {"Gérald Schwartzmann", NULL};
/*	char * documentors[] = {NULL};	*/
	char license[] = "This program is free software; you can redistribute it and/or\n" 
		"modify it under the terms of the GNU General Public License as\n"
 		"published by the Free Software Foundation; either version 2 of the\n"
		"License, or (at your option) any later version.\n\n"
		"This program is distributed in the hope that it will be useful,\n"
		"but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
		"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
		"GNU General Public License for more details.\n\n"
		"You should have received a copy of the GNU General Public License\n"
		"along with this program; if not, write to the Free Software\n"
		"Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02111-1307\n"
		"USA";
/* Note to translators: replace the following string with the appropriate credits for you lang */
	char *translator_credits = _("translator_credits");
	GdkPixbuf* pixbuf = gdk_pixbuf_new_from_file ( DATADIR"/"PACKAGE"/yogi.jpg", NULL);
	gtk_show_about_dialog (NULL,
					"name", "Yogi",
					"authors", authors,
					"artists", artists,
					"comments", _("Yogi is a GTK based graphical front end to the image corrector clens."),
					"copyright", _("(C) 2006 by Jean Bréfort"),
					"license", license,
					"logo", pixbuf,
/*					"icon-name", "yogi",*/
					"translator_credits", strcmp (translator_credits, "translator_credits") != 0 ? 
											(const char *)translator_credits : NULL,
					"version", VERSION,
					"website", "http://yogi.nongnu.org",
					NULL);
	if (pixbuf != NULL)
		g_object_unref (pixbuf);
}

void prefs_clicked_cb ()
{
	GladeXML *xml;

	xml = glade_xml_new (DATADIR"/yogi/prefs.glade", "prefs", NULL);
	glade_xml_signal_autoconnect (xml);
	profile_sel = GTK_FILE_CHOOSER (glade_xml_get_widget (xml, "profiles"));
	gtk_file_chooser_set_filename (profile_sel, profile_path);
	stitcher_sel = GTK_FILE_CHOOSER (glade_xml_get_widget (xml, "stitcher"));
	gtk_file_chooser_set_filename (stitcher_sel, stitcher);
	prefs = glade_xml_get_widget (xml, "prefs");
}

void prefs_apply_clicked_cb ()
{
	g_free (profile_path);
	profile_path = g_strdup (gtk_file_chooser_get_filename (profile_sel));
	gconf_client_set_string (conf_client, "/apps/yogi/profile-path", profile_path, NULL);
	g_free (stitcher);
	stitcher = g_strdup (gtk_file_chooser_get_filename (stitcher_sel));
	gconf_client_set_string (conf_client, "/apps/yogi/stitcher", stitcher, NULL);
}

void prefs_ok_clicked_cb ()
{
	prefs_apply_clicked_cb ();
	gtk_widget_destroy (prefs);
}

void prefs_cancel_clicked_cb ()
{
	gtk_widget_destroy (prefs);
}

void file_activated_cb (GtkFileChooser *file)
{
	GError *error = NULL;
	char *command, *output = NULL;
	char *camera = NULL, *lens;
	char **lenses;
	int i, result;
	filename = gtk_file_chooser_get_filename (file);
	GdkPixbuf *pixbuf = gdk_pixbuf_new_from_file_at_scale (filename,
										width, height, TRUE, &error);
	gtk_image_set_from_pixbuf (source, pixbuf);
	command = g_strdup_printf ("clens -p %s/profile.txt --Lenses %s",
		profile_path, filename);
	if (!g_spawn_command_line_sync (command, &output, NULL, &result, &error)) {
		if (error) {
			g_warning (error->message);
			g_error_free (error);
		}
		if (output)
			g_free (output);
		g_free (command);
		return;
	}
	g_free (command);
	lenses = g_strsplit (output, "\n", 0);
	g_free (output);
	gtk_list_store_clear (lenses_list);
	if (!strcmp (lenses[1] + 1, "Exif read failed")) {
		g_warning ("No exif data in image");
	} else if (strncmp (lenses[1] + 1, "Camera ", 6)) {
		g_warning ("Error while executing clens: %s",lenses[1]);
	} else {
		GtkTreeIter iter;
		camera = g_strdup (lenses[1] + 9);
		gtk_label_set_text (camera_lbl, camera);
		gtk_list_store_clear (lenses_list);
		i = 2;
		while (lenses[i] && *lenses[i] == '\t')
			gtk_combo_box_append_text (lenses_box, lenses[i++] + 1);
		gtk_combo_box_set_active (lenses_box, 0);
	}
	g_strfreev (lenses);
}

void execute_cb ()
{
	char *command, *output = NULL, *lens = NULL;
	char const *ext, *ext2;
	GString *cmdstr;
	GError *error = NULL;
	GtkTreeIter iter;
	int l, quality;
	gboolean tiff, unscale;
	if (filename == NULL)
		return;
	gtk_combo_box_get_active_iter (lenses_box, &iter);
	gtk_tree_model_get (GTK_TREE_MODEL (lenses_list), &iter, 0, &lens, -1);
	if (!lens || !(*lens))
		return;
	gdk_window_set_cursor (main_win, cursor);
	while (gtk_events_pending ())
		gtk_main_iteration ();
	cmdstr = g_string_new ("clens");
	ext = gtk_entry_get_text (ext_entry);
	if (ext && *ext)
		g_string_append_printf (cmdstr, " -a \"%s\"", ext);
	else
		ext = "_pt";
	unscale = gtk_toggle_button_get_active (unscale_btn);
	if (unscale)
		g_string_append (cmdstr, " %u");
	tiff = gtk_toggle_button_get_active (tiff_btn);
	if (tiff) {
		g_string_append (cmdstr, " --tiff");
		ext2 = ".tif";
	} else {
		quality = gtk_spin_button_get_value_as_int (jpeg_btn);
		if (quality != 90)
			g_string_append_printf (cmdstr, " -q \"%d\"", quality);
		ext2 = ".jpg";
	}
	g_string_append_printf (cmdstr, " -p %s/profile.txt -s %s -l \"%s\" %s",
		profile_path, stitcher, lens, filename);
	command = g_string_free (cmdstr, FALSE);
	if (!g_spawn_command_line_sync (command, &output, NULL, NULL, &error)) {
		if (error) {
			g_warning (error->message);
			g_error_free (error);
		}
		if (output)
			g_free (output);
		gdk_window_set_cursor (main_win, NULL);
		return;
	}
	g_free (output);
	g_free (command);
	command = g_strdup (filename);
	output = command + strlen (filename);
	while (output > command && *output != '.')
		output--;
	gdk_window_set_cursor (main_win, NULL);
	if (output == command)
		return;
	*output = 0;
	lens = g_strconcat (command, ext, ext2, NULL);
	g_free (command);
	GdkPixbuf *pixbuf = gdk_pixbuf_new_from_file_at_scale (lens,
										width, height, TRUE, &error);
	gtk_image_set_from_pixbuf (dest, pixbuf);
	g_free (lens);
}

int main (int argc, char *argv[])
{
	GladeXML *xml;
	GtkWidget *w;
	GtkFileFilter* filter;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GError *error = NULL;

	gtk_init (&argc, &argv);
	xml = glade_xml_new (DATADIR"/yogi/yogi.glade", "yogi", NULL);
	glade_xml_signal_autoconnect (xml);
	filter = gtk_file_filter_new ();
	gtk_file_filter_add_mime_type (filter, "image/jpeg");
	w = glade_xml_get_widget (xml, "filename");
	gtk_file_chooser_add_filter (GTK_FILE_CHOOSER (w), filter);
	source = GTK_IMAGE (glade_xml_get_widget (xml, "image1"));
	dest = GTK_IMAGE (glade_xml_get_widget (xml, "image2"));
	camera_lbl = GTK_LABEL (glade_xml_get_widget (xml, "camera"));
	lenses_box = GTK_COMBO_BOX (glade_xml_get_widget (xml, "lenses"));
	lenses_list = GTK_LIST_STORE (gtk_combo_box_get_model (lenses_box));
	unscale_btn = GTK_TOGGLE_BUTTON (glade_xml_get_widget (xml, "no-dim"));
	tiff_btn = GTK_TOGGLE_BUTTON (glade_xml_get_widget (xml, "tiff"));
	ext_entry = GTK_ENTRY (glade_xml_get_widget (xml, "ext"));
	jpeg_btn = GTK_SPIN_BUTTON (glade_xml_get_widget (xml, "jpeg"));
	cursor = gdk_cursor_new (GDK_WATCH);
	w = glade_xml_get_widget (xml, "yogi");
	main_win = w->window;
	gtk_combo_box_set_active (lenses_box, 0);

	conf_client = gconf_client_get_default ();
	gconf_client_add_dir (conf_client, "/apps/yogi", GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);
	profile_path = gconf_client_get_string (conf_client, "/apps/yogi/profile-path", &error);
	if (error) {
		g_message ("GConf failed: %s", error->message);
		g_error_free (error);
		error = NULL;
	}
	if (!profile_path)
		profile_path = g_strdup ("/usr/share/clens/");
	stitcher = gconf_client_get_string (conf_client, "/apps/yogi/stitcher", &error);
	if (error) {
		stitcher = g_strdup ("/usr/bin/nona");
		g_message ("GConf failed: %s", error->message);
		g_error_free (error);
		error = NULL;
	}
	if (!stitcher)
		stitcher = g_strdup ("/usr/bin/nona");
	notification_id = gconf_client_notify_add (conf_client, "/apps/yogi", config_changed_cb, NULL, NULL, NULL);

	gtk_main ();

	g_free (profile_path);
	g_free (stitcher);
	gconf_client_notify_remove (conf_client, notification_id);
	gconf_client_remove_dir (conf_client, "/apps/yogi", NULL);
	g_object_unref (G_OBJECT (conf_client));
	return 0;
}
