///////////////////////////////////////////////////////////////////////////////
// Copyright (C) 2008 Rene Jensen                                            //
//                                                                           //
// This file is part of YUV4MPEG Motion Tools (YUVMotionTools).              //
//                                                                           //
// Authors: Rene Jensen <centipede@takhis.net>                               //
//                                                                           //
// YUVMotionTools is free software: you can redistribute it and/or modify    //
// it under the terms of the GNU General Public License as published by      //
// the Free Software Foundation, either version 3 of the License, or         //
// (at your option) any later version.                                       //
//                                                                           //
// YUVMotionTools is distributed in the hope that it will be useful,         //
// but WITHOUT ANY WARRANTY; without even the implied warranty of            //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             //
// GNU General Public License for more details.                              //
//                                                                           //
// You should have received a copy of the GNU General Public License         //
// along with YUVMotionTools.  If not, see <http://www.gnu.org/licenses/>.   //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include "featurepoint.h"

FeaturePoint::FeaturePoint (const CvPoint2D32f& startPoint, int firstframe)
{
    history.push_back (startPoint);
    firstFrame = firstframe;
    lastFrame = firstframe;
    lifecycle = ALIVE;
}

points_t::iterator  FeaturePoint::getPointsBeginIt ()    { return history.begin(); }
points_t::iterator  FeaturePoint::getPointsEndIt   ()    { return history.end(); }
int                 FeaturePoint::getPointCount    ()    { return history.size(); }
status_e            FeaturePoint::getStatus        ()    { return lifecycle; }
int                 FeaturePoint::getFirstFrame    ()    { return firstFrame; }
int                 FeaturePoint::getLastFrame     ()    { return lastFrame; }
CvPoint2D32f        FeaturePoint::getLatestPos     ()    { return history.back(); }
CvPoint             FeaturePoint::getPoint         ()    { return cvPoint (history.back().x , history.back().y); }

CvPoint2D32f FeaturePoint::getPosition (int frame)
{
    if (frame < 0)
    {
        if (-frame > history.size())
            return cvPoint2D32f (0,0);
        return history.end()[frame];

    }
    else
    {
        if (frame < firstFrame || frame >= firstFrame+history.size())
            return cvPoint2D32f (0,0);

        return history[frame - firstFrame];
    }
}


CvPoint2D32f FeaturePoint::getVelocity (int frame)
{
    CvPoint2D32f b = getPosition(frame);
    CvPoint2D32f a = getPosition(frame-1);
    return cvPoint2D32f (b.x-a.x , b.y-a.y);

/*    if (history.size() < 2)
        return cvPoint2D32f (0,0);
    CvPoint2D32f& b = *(history.end()-1);
    CvPoint2D32f& a = *(history.end()-2);
    return cvPoint2D32f (b.x-a.x , b.y-a.y);*/
}

CvPoint2D32f FeaturePoint::getAcceleration (int frame)
{
    CvPoint2D32f b = getVelocity(frame);
    CvPoint2D32f a = getVelocity(frame-1);
    return cvPoint2D32f (b.x-a.x , b.y-a.y);
/*    if (history.size() < 2)
        return cvPoint2D32f (0,0);
    CvPoint2D32f& b = *(history.end()-1);
    CvPoint2D32f& a = *(history.end()-2);
    return cvPoint2D32f (b.x-a.x , b.y-a.y);*/
}

CvScalar FeaturePoint::getColor ()
{
    //int st = int (track_error[i] * 255);
    int r = int (error_x * 1);
    int g = int (error_y * 1);
    int b = 0;//status[i]*255;
    return CV_RGB(r,g,b);
}

void FeaturePoint::updateHistory (const CvPoint2D32f& pos, char status, float errorx, float errory)
{
    history.push_back (pos);
    error_x = errorx;
    error_y = errory;
    if (status == 0 || error_x > 500 || error_y > 500)
    {
        lifecycle = DEAD;
    }
}

void FeaturePoint::updateDeltas (int currentFrame, float& dX, float& dY, float& weight)
{
    if (lifecycle == ALIVE && history.size() > 1)
    {
        weight += 1;
        dX += history.end()[-1].x - history.end()[-2].x;
        dY += history.end()[-1].y - history.end()[-2].y;
    }
}

void FeaturePoint::trackPoints (
    featurepoints_t& points,
    IplImage*        oldImage,
    IplImage*        newImage,
    IplImage*        oldPyramidTmp,
    IplImage*        newPyramidTmp)
{
    static CvPoint2D32f * next = 0;
    static CvPoint2D32f * prev = 0;
    static char *         status = 0;
    static float *        error = 0;
    static int            capacity = 0;

    int count = points.size();

    // Static init: Create the work images as needed
    if (capacity < count)
    {
        if (capacity > 0) {
            delete[] next;
            delete[] prev;
            delete[] status;
            delete[] error;
        }
        capacity = count + 200;
        next   = new CvPoint2D32f[capacity];
        prev   = new CvPoint2D32f[capacity];
        status = new char[capacity];
        error  = new float[capacity*2];
    }


    // Copy to temp buffer
    int i=0;
    for (featurepoints_t::iterator I=points.begin(); I!=points.end(); ++I) {
        prev[i] = ((*I)->getLatestPos());
        ++i;
    }
    // Track flow
    cvCalcOpticalFlowPyrLK (
        oldImage, newImage, oldPyramidTmp, newPyramidTmp, prev, next, count,
        cvSize(10,10), 3, status, error, cvTermCriteria (CV_TERMCRIT_ITER|CV_TERMCRIT_EPS, 20, 0.03), 0
    );

    // Copy back to points
    i=0;
    for (featurepoints_t::iterator I=points.begin(); I!=points.end(); ++I) {
        (*I)->updateHistory (next[i], status[i], error[i*2], error[i*2+1]);
        ++i;
    }
}


featurepoints_t FeaturePoint::findNew (
    int             howMany,
    IplImage*       image,
    int             frameStart,
    const CvSize&   frameSize)
{
    static IplImage*        tmpEig = 0;
    static IplImage*        tmpAux = 0;
    static CvPoint2D32f *   corners = 0;
    static int              count = 0;
    // Static init: Create the work images as needed
    if (! tmpEig)
    {
        tmpEig = cvCreateImage (frameSize, IPL_DEPTH_32F, 1);
        tmpAux = cvCreateImage (frameSize, IPL_DEPTH_32F, 1);
        assert (tmpEig!=0);
        assert (tmpAux!=0);

        corners = new CvPoint2D32f[2000];
    }
    count = howMany;
    cvGoodFeaturesToTrack (image, tmpEig, tmpAux, corners, &count, 0.01, 10, 0, 3,0,.04);
    cvFindCornerSubPix (image, corners, count, cvSize(11,11), cvSize(-1,-1), cvTermCriteria(CV_TERMCRIT_ITER|CV_TERMCRIT_EPS, 20, 0.01));

    featurepoints_t points;
    for (int i=0; i<count; ++i)
        points.push_back (new FeaturePoint (corners[i], frameStart));

    return points;
}
