///////////////////////////////////////////////////////////////////////////////
// Copyright (C) 2008 Rene Jensen                                            //
//                                                                           //
// This file is part of YUV4MPEG Motion Tools (YUVMotionTools).              //
//                                                                           //
// Authors: Rene Jensen <centipede@takhis.net>                               //
//                                                                           //
// YUVMotionTools is free software: you can redistribute it and/or modify    //
// it under the terms of the GNU General Public License as published by      //
// the Free Software Foundation, either version 3 of the License, or         //
// (at your option) any later version.                                       //
//                                                                           //
// YUVMotionTools is distributed in the hope that it will be useful,         //
// but WITHOUT ANY WARRANTY; without even the implied warranty of            //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             //
// GNU General Public License for more details.                              //
//                                                                           //
// You should have received a copy of the GNU General Public License         //
// along with YUVMotionTools.  If not, see <http://www.gnu.org/licenses/>.   //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include "trackingbuffers.h"

TrackingBuffers::TrackingBuffers (int width, int height, int subsampleX, int subsampleY)
{
    CvSize subs;

    usingXformed = false;
    frontIsOne = true;
    size = cvSize (width, height);
    subs = cvSize (width/subsampleX, height/subsampleY);
    counter = 0;

    luma1 = cvCreateImageHeader (size, IPL_DEPTH_8U, 1);
    luma2 = cvCreateImageHeader (size, IPL_DEPTH_8U, 1);
    luma1->imageData = new char [width * height];
    luma2->imageData = new char [width * height];

    chroma1 = cvCreateImage (size, IPL_DEPTH_8U, 1);
    chroma2 = cvCreateImage (size, IPL_DEPTH_8U, 1);
    imageYUV = cvCreateImage (size, IPL_DEPTH_8U, 3);
    imageRGB = cvCreateImage (size, IPL_DEPTH_8U, 3);
    imageRGBxform = cvCreateImage (size, IPL_DEPTH_8U, 3);

    chromaSubs1 = cvCreateImageHeader (subs, IPL_DEPTH_8U, 1);
    chromaSubs2 = cvCreateImageHeader (subs, IPL_DEPTH_8U, 1);
    chromaSubs1->imageData = new char [width/subsampleX * height/subsampleY];
    chromaSubs2->imageData = new char [width/subsampleX * height/subsampleY];

    pyramid1 = cvCreateImage(size, IPL_DEPTH_8U, 1);
    pyramid2 = cvCreateImage(size, IPL_DEPTH_8U, 1);
}

TrackingBuffers::TrackingBuffers ()
{
    // Delete buffers
}

void        TrackingBuffers::flip ()             { frontIsOne = ! frontIsOne; }
CvSize&     TrackingBuffers::getSize ()          { return size; }
int         TrackingBuffers::getCounter ()       { return counter; }
IplImage*   TrackingBuffers::getImageYUV  ()     { return imageYUV; }
IplImage*   TrackingBuffers::getImageRGB  ()     { return usingXformed ? imageRGBxform : imageRGB; }
IplImage*   TrackingBuffers::getFrontLuma    ()  { return frontIsOne ? luma1 : luma2; }
IplImage*   TrackingBuffers::getBackLuma     ()  { return frontIsOne ? luma2 : luma1; }
IplImage*   TrackingBuffers::getFrontPyramid ()  { return frontIsOne ? pyramid1 : pyramid2; }
IplImage*   TrackingBuffers::getBackPyramid  ()  { return frontIsOne ? pyramid2 : pyramid1; }

void TrackingBuffers::getBuffers (uint8_t*& luma, uint8_t*& chroma1, uint8_t*& chroma2)
{
    luma    = (uint8_t*) getFrontLuma()->imageData;
    chroma1 = (uint8_t*) chromaSubs1->imageData;
    chroma2 = (uint8_t*) chromaSubs2->imageData;
}

void TrackingBuffers::prepareForRead ()
{
    cvResize (chromaSubs1, chroma1, CV_INTER_LINEAR);
    cvResize (chromaSubs2, chroma2, CV_INTER_LINEAR);
    cvMerge (getFrontLuma(), chroma1, chroma2, 0, imageYUV);
    cvCvtColor (imageYUV, imageRGB, CV_YCrCb2RGB);

    usingXformed = false;
    ++ counter;
}


void TrackingBuffers::prepareForWrite ()
{
    cvCvtColor (getImageRGB(), imageYUV, CV_RGB2YCrCb);
    cvSplit (imageYUV, getBackLuma(), chroma1, chroma2, 0);
    cvResize (chroma1, chromaSubs1, CV_INTER_LINEAR);
    cvResize (chroma2, chromaSubs2, CV_INTER_LINEAR);

    flip();
}


void TrackingBuffers::translateRGB (float Tx, float Ty)
{
    CvMat R = cvMat (2,3, CV_32FC1);
    cvCreateData (&R);
    //static float angle = 0.0;
    //cv2DRotationMatrix (cvPoint2D32f(720/2,576/2), angle, 1, &R);
    //angle += 0.1;
    R.data.fl[0] = 1.0;
    R.data.fl[1] = 0.0;
    R.data.fl[2] = Tx;
    R.data.fl[3] = 0.0;
    R.data.fl[4] = 1.0;
    R.data.fl[5] = Ty;
    cvWarpAffine (imageRGB, imageRGBxform, &R);
    usingXformed = true;
}
