##############################################################################
=pod

=head1 NAME

 zebot::DB::Helper

=head1 DESCRIPTION

zebot::DB::Helper module, fires up a number of worker processes that connect to the database, and then forks off with the incoming tasks.

=head1 COPYRIGHT and LICENCE

  Copyright (c) 2002 Bruno Boettcher

  DB::Helper.pm is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; version 2
  of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

=head1 Methods of this class

=over

=cut

##############################################################################
package zebot::DB::Helper;
use strict;
use POSIX qw(strftime);
use Data::Dumper;

our ($statements,$stindex);


##############################################################################
=pod

=item CTOR

instantiation of the Helper

=cut

##############################################################################
sub new {
  my $this = bless {}, "zebot::DB::Helper";
  return $this;
}
##############################################################################
=pod

=item init

Initiaize, means set up this module

=cut

######################################################################
sub init
{
  my ($this, $database,$dbuser,$dbpasswd) = @_;

  #$this->print("connecting to DB with $database,$dbuser,$dbpasswd!\n");
  my $dbh = DBI->connect($database,$dbuser,$dbpasswd,
      {AutoCommit => 0,PrintError =>0})
  or die("DB::Helper: Couldn't connect to database: ".DBI->errstr."\n");
  #$this->print("CONNECTED TO DB=$dbh\n");
  $this->dbh($dbh);

}#sub init
######################################################################
=pod

=item dbh

attribute getter previously provided by ObjecTemplate

=cut

######################################################################
sub dbh
{
  my($this,$ldbh) = @_;

  if($ldbh)
  {
    $this->{"dbh"} = $ldbh;
  }
  return $this->{"dbh"};
}# sub dbh
#########################################################
=pod

=item print

delegator, to make a copy of the console stream to a file, for debugging
purposes only, since somehow the STDOUT gets swallowed up for some cases....

=cut

#########################################################
sub print
{
  my ($this,$msg) = @_;
  $msg = $this if(!$msg);
  if(open(FILE,">>logger"))
  {
    print FILE ("$msg\n");
    close(FILE);
  }# if(open(FILE,">logger"))
  #print(STDERR "$msg\n");
}# sub$this->print
######################################################################
=pod

=item version

return the version of this module

=cut

######################################################################
sub version
{
  return '$Revision: 1.7 $';
}#sub isa

######################################################################
=pod

=item doQuery

perform a query to the database
arguments is hash:
query: SQL string or prepared statemnt
values: the optional arguments
type: hash|count for the momet for the returntype


=cut

######################################################################
sub doQuery
{
  my ($this, $args) = @_;
  # $this->print("--------------- actquery='$sql_query'\n");
  $this->{"running"} = 1;
  #$this->print("DOQUERY ".Dumper($args)." \n");

  my $sth;
  my $dbh = $this->dbh();

  my $query = $args->{"query"};
  #$this->print("DOQUERY trying to perform $query\n");

  if($query =~ /^STH/)
  {
    #$this->print("DOQUERY STH type\n");
    $sth = $statements->{$query};
    if(!$sth)
    {
      #$this->print("DOQUERY CRETIN pas de STH??\n");
      $dbh->rollback(); 
      $this->{"running"} = 0;
      $args->{"values"} = "unknown/non prepared query $query"; 
      $args->{"status"} = "failure";
      return $this->doReturn($args);
    }
  }# if($query =~ /^STH/)
  else
  {
    if(!($sth = $dbh->prepare($query)))
    {
      $this->print("DOQUERY pas moyen de crer STH??\n");
      $dbh->rollback(); 
      $this->{"running"} = 0;
      $args->{"values"} = $sth->errs; 
      $args->{"status"} = "failure";
      return $this->doReturn($args);
    }# if(!($sth = $dbh->prepare($sql_query)))
  }# else

  $this->{"running"} = 0;
  $args = $this->doExecute($sth, $args);
  #$this->print("DOQUERY so far successful $query returning ".Dumper($args)."\n");

  #$this->print("Helper::doQuery got result: ".Dumper($args));
  return $this->doReturn($args);
}#sub doQuery    
######################################################################
=pod

=item doExecute

execute a prepared statement
the return value is allready stripped of all DBI interference

=cut

######################################################################
sub doExecute
{
  my ($this, $sth, $args) = @_;

  my $dbh = $this->dbh();

  my $success;
  #$this->print("doExecute with ".Dumper($args->{"values"}));
  if($args->{"values"})
  {
    $success = $sth->execute(@{$args->{"values"}});
  }# if($args->{"$values"})
  else
  {
    $success = $sth->execute();
  }
  #$this->print("DOEXEapres exe avec : ".Dumper($success)." et ".$sth->errstr.", ".$sth->state.", ".$DBI::errstr."\n");
  if($success)             # Execute the query
  {
    $dbh->commit() if(!($args->{"commit"}));
    my $data = $this->condense($sth,$args);

    $args->{"values"} = $data; 
    $args->{"status"} = "success";
    #$this->print("DOEXEapres SUCCESS with $data\n");
    return($args);
  }
  else
  {
    $dbh->rollback(); 
    $args->{"values"} = $sth->errstr; 
    $args->{"status"} = "failure";
    $this->print("DOEXEapres FAILURE with ".$sth->errstr."\n");
    return($args);
  }
}# sub doExecute
######################################################################
=pod

=item condense

pack up the data we received from the DB
array of hashes

=cut

######################################################################
sub condense
{
  my ($this,$sth, $args) = @_;

  #$this->print("DOCONDENSE type=".$args->{"type"}." \n");

  if($args->{"type"} =~ /hash/i)
  {
    my @toReturn = ();
    while(my $line = $sth->fetchrow_hashref())
    {
      push(@toReturn,$line);
    }# while(my $line = $sth->fetchrow_hashref())
    return \@toReturn;
  }# if($args->"type" =~ /hash/i)
  elsif($args->{"type"} =~ /count/i)
  {
    my @toReturn = [$sth->rows];
    return \@toReturn;
  }# if($args->"type" =~ /hash/i)
  return;
}#sub condense
######################################################################
=pod

=item isa

return the type of thie module

=cut

######################################################################
sub isa
{
  return "DB::Helper";
}#sub isa
######################################################################
=pod

=item shutdown

close down the activity for a safe shutdown

=cut

######################################################################
sub shutdown
{
  my $this = shift;
  my $dbh = $this->dbh();
  $dbh->disconnect();
}#sub shutdown
######################################################################
=pod

=item prepare

prepare a sql statement for further refinement

=cut

######################################################################
sub prepare
{
  my ($this, $args) = @_;
  my $sql_query;
  if(ref($args) eq 'ARRAY')
  {
    $sql_query = $args->{"query"};
  }# if(ref($args) eq 'ARRAY')
  else
  {
    $sql_query = $args;
    $args = {};
  }# else
  my $dbh = $this->dbh();
  #$this->print("IS CONNECTED TO DB=$dbh\n");
  my $sth;
  $this->{"running"} = 1;

  #$this->print("trying to prepare $sql_query\n");
  $sth = $dbh->prepare($sql_query);
  #$this->print("after preparing $sth\n");

  if(!$sth)
  { 
    $dbh->rollback(); 
    $this->{"running"} = 0;
    $args->{"values"} = $dbh->errstr; 
    $args->{"status"} = "failure";
  #$this->print("failed to prepare $sql_query\n");
    return $this->doReturn($args);
  };
  #$this->print("success preparing  $sql_query\n");
  $stindex++;
  $statements->{"STH$stindex"} = $sth;
  $this->{"running"} = 0;
  $args->{"values"} = "STH$stindex"; 
  $args->{"status"} = "success";
  #print("should return structure:".Dumper($args)."\n");
  #my $retVal = $this->doReturn($args);
  my $retVal = $args;
  #print("returning:$retVal\n");
  return $retVal;
}# sub prepare
######################################################################
=pod

=item free

drop a prepared statement to liberate some space...

=cut

######################################################################
sub free
{
  my ($this, $identifier) = @_;
  delete $statements->{$identifier};
}#sub shutdown
######################################################################
=pod

=item batch

process a whole transaction
arguments are 1 arrays of queries as described in doQuery

=cut

######################################################################
sub batch
{
  my ($this, $job) = @_;
  my $queries = $job->{"query"};
  my $pbsession = $job->{"session"};
  my $pbevent = $job->{"event"};

  my $workers = $this->{"workers"};
  my $tasks = $this->{"tasks"};
  my $batch = $this->{"batch"};
  $this->{"running"} = 1;

  my $msg  = "received batch: ".Dumper( $job)."\n";
  $this->print("BEGIN OF Helper BATCH for ".Dumper($queries)."\n");

  # save the mode we are running in, in the meantime switch to ref mode
  my $modebackup = $this->{"resultAsReference"};
  $this->{"resultAsReference"} = 0;

  my $subjob;
  %$subjob = %$job;

  #devalidate autocommit
  $subjob->{"commit"} = 1;

  my $toReturn = ();
  foreach my $line (@$queries)
  {
    my $sth;
    if(ref($line) ne 'HASH')
    {
      #we have a simple SQL query here, wrap it!
      $subjob = {}; 
      foreach my $key (%$job)
      {
	$subjob->{$key} = $job->{$key};
      }
	$subjob->{"query"} = $line;
    }# if(ref($line) ne 'HASH')
    else
    {
      #$subjob->{"query"} = $line;
      $subjob = $line;
    }# else

    $subjob->{"byref"} = 1;
    $msg .= "issuing batch: line ".Dumper( $subjob)."\n";
    $this->print("QUERY: ".Dumper( $subjob)."\n");
    my $returns = $this->doQuery($subjob);
    $this->{"running"} = 1;
    $this->print("ISSUED ".Dumper($subjob)." got ".Dumper($returns)." \n");

    if($returns->{"status"} eq "success" )
    {
      push(@$toReturn,$returns);
    }# if($returns->{"status"} eq "success" )
    else
    {
      $this->{"running"} = 0;
      $subjob->{"values"} = $returns->{"values"}; 
      $subjob->{"status"} = "failure";
      $this->{"resultAsReference"} = $modebackup;
      $this->{"debug"} .=  $msg;
      $this->print("NOSUCCESS::".$msg);
      return $this->doReturn($subjob);
    }# else
  }# foreach my $query (@$queries)

  #seems a success, do commit now
  my $dbh = $this->dbh();
  $dbh->commit();
  $this->print("COMMITTED!!\n");
  $this->{"running"} = 0;
  #use the last line as sample...
  $subjob->{"values"} = $toReturn; 
  $subjob->{"status"} = "success";
  $this->{"resultAsReference"} = $modebackup;
  $this->{"debug"} .=  $msg;
  #$this->print("SUCCESS::".$msg);
  return $this->doReturn($subjob);
}#sub batch
######################################################################
=pod

=item doReturn

since the return stuff grew complicated, made an own return method..
check if we can return a perl reference or if it must be go through a packing and to storable conversion stage

=cut

######################################################################
sub doReturn
{
  my($this, $args) = @_;

  my $cond = $this->{"resultAsReference"};
  if( $args->{"byref"})
  {
    $cond = 0;
  }# if( $args->{"byref"})

  if($cond)
  {
    $args->{"session"} = $this->{"session"};
    $args->{"event"} = $this->{"event"};

    my $filter = POE::Filter::Reference->new();
    my $output = $filter->put([ $args ] );
    #$this->print("backupdata OUTPUT: ".Dumper($this->{"result"}));
    print STDOUT @$output;
    return;
  }# if($this->{"resultAsReference"})
  else
  {
    return($args);
  }#else
}#sub return
1
__END__

=back

=head1 AUTHOR

Bruno Bttcher <bboett at adlp.org>

=head1 SEE ALSO

zebot home page  http://www.freesoftware.fsf.org/zebot/ 
POD documentation of zebot

=cut

