/*  Ztest - verify integrity of compressed files
    Copyright (C) 2010, 2011, 2012, 2013 Antonio Diaz Diaz.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

void show_ztest_help()
  {
  std::printf( "Ztest verifies the integrity of the specified compressed files.\n"
               "Non-compressed files are ignored. If no files are specified, the\n"
               "integrity of compressed data read from standard input is verified. Data\n"
               "read from standard input must be all in the same compression format.\n"
               "\nThe supported formats are bzip2, gzip, lzip and xz.\n"
               "\nNote that some xz files lack integrity information, and therefore can't\n"
               "be verified as reliably as the other formats can.\n"
               "\nUsage: ztest [options] [files]\n"
               "\nExit status is 0 if all compressed files verify OK, 1 if environmental\n"
               "problems (file not found, invalid flags, I/O errors, etc), 2 if any\n"
               "compressed file is corrupt or invalid.\n"
               "\nOptions:\n"
               "  -h, --help                 display this help and exit\n"
               "  -V, --version              output version information and exit\n"
               "      --format=<fmt>         force given format (bz2, gz, lz, xz)\n"
               "  -q, --quiet                suppress all messages\n"
               "  -r, --recursive            operate recursively on directories\n"
               "  -v, --verbose              be verbose (a 2nd -v gives more)\n" );
  show_help_addr();
  }


int ztest_stdin( const int infd, const int format_type,
                 const std::vector< const char * > & ztest_args )
  {
  const uint8_t * magic_data = 0;
  int magic_size = 0;
  const char * const decompressor_name = ( format_type >= 0 ) ?
    decompressor_names[format_type] :
    test_format( infd, &magic_data, &magic_size );
  if( !decompressor_name )
    { show_error( "Unknown data format read from stdin." ); return 2; }
  int fda[2];				// pipe from feeder
  if( pipe( fda ) < 0 )
    { show_error( "Can't create pipe", errno ); return 1; }

  const pid_t pid = fork();
  if( pid == 0 )			// child1 (decompressor)
    {
    if( dup2( fda[0], STDIN_FILENO ) >= 0 &&
        close( fda[0] ) == 0 && close( fda[1] ) == 0 )
      {
      const char ** const argv = new const char *[ztest_args.size()+3];
      argv[0] = decompressor_name;
      for( unsigned i = 0; i < ztest_args.size(); ++i )
        argv[i+1] = ztest_args[i];
      argv[ztest_args.size()+1] = "-t";
      argv[ztest_args.size()+2] = 0;
      execvp( argv[0], (char **)argv );
      }
    show_exec_error( decompressor_name );
    _exit( 1 );
    }
					// parent
  if( pid < 0 )
    { show_fork_error( decompressor_name ); return 1; }

  const pid_t pid2 = fork();
  if( pid2 == 0 )			// child2 (decompressor feeder)
    {
    if( close( fda[0] ) != 0 ||
        !feed_data( infd, fda[1], magic_data, magic_size ) )
      _exit( 1 );
    if( close( fda[1] ) != 0 )
      { show_close_error( "decompressor feeder" ); _exit( 1 ); }
    _exit( 0 );
    }
					// parent
  if( pid2 < 0 )
    { show_fork_error( "decompressor feeder" ); return 1; }

  close( fda[0] ); close( fda[1] );
  int retval = wait_for_child( pid, decompressor_name, 1 );
  if( retval == 0 && wait_for_child( pid2, "decompressor feeder" ) != 0 )
    retval = 1;
  return retval;
  }


int ztest_file( const int infd, const int format_type,
                const std::string & input_filename,
                const std::vector< const char * > & ztest_args )
  {
  const uint8_t * magic_data = 0;
  int magic_size = 0;
  const char * const decompressor_name = ( format_type >= 0 ) ?
    decompressor_names[format_type] :
    test_format( infd, &magic_data, &magic_size );
  if( !decompressor_name )
    return 0;				// skip this file
  const pid_t pid = fork();

  if( pid == 0 )			// child (decompressor)
    {
    const char ** const argv = new const char *[ztest_args.size()+5];
    argv[0] = decompressor_name;
    for( unsigned i = 0; i < ztest_args.size(); ++i )
      argv[i+1] = ztest_args[i];
    argv[ztest_args.size()+1] = "-t";
    argv[ztest_args.size()+2] = "--";
    argv[ztest_args.size()+3] = input_filename.c_str();
    argv[ztest_args.size()+4] = 0;
    execvp( argv[0], (char **)argv );
    show_exec_error( decompressor_name );
    _exit( 1 );
    }
					// parent
  if( pid < 0 )
    { show_fork_error( decompressor_name ); return 1; }

  return wait_for_child( pid, decompressor_name, 1 );
  }
